#
# annealing_find_ecology_parameters.R
#
#' Optimise StrathE2E ecology model parameters to maximise the likelihood of observed ecosystem target data
#'
#' Launches a StrathE2E simulated annealing process to find the set of ecology model parameters
#' producing the maximum likelihood of observed target data on the state of the ecosystem,
#' given specified environmental driving data and fishing fleet parameters.
#'
#' Simulated annealing is an iterative random-walk type process which searches the parameter space
#' of a model to locate the combination which maximises the likelihood of a set of observed
#' data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
#' may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
#' avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
#' by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1)
#'
#' Model configuration and initial values of the ecology model parameters need to be
#' assembled by a prior call of the read_model() function.
#' 
#' NOTE that the user.path argument in the read_model() function call needs to point to a user workspace folder, not the default
#' North Sea model provided with the package. This is because the annealing function needs write-access to the model /Parameters folder,
#' but the /extdata/Models folder in the package installation is read-only.
#' To use the annealing function on the North Sea model, use the copy_model() function to make a copy of the
#' North Sea model in the user workspace.
#'
#' The coefficients of variation for jiggling the ecology parameter can be varied in real-time
#' during the run by editing the file "annealing_SD_ecology.csv" in the folder
#' /Parameters/Parameter_SD_control/ of the model version.
#'
#' The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
#' the likelihoods of the proposed and accepted parameter sets at each iteration.
#' Y-axis (likelihood o fthe target data) range of the real time plot can be varied during the run by 
#' editing the setup file "annealing_SD_ecology.csv"
#'
#' At the end of the procedure new versions of the three ecology model fitted_parameters.. files are exported
#' to the folder /Parameters of the model version, with a user defined identifier specified by the model.ident argument
#' in the read_model() function. The histories of proposed and accepted parameter combinations
#' are saved as CSV files in the results folder.
#'
#' In order to use the new fitted parameter values in a subsequent run of the StrathE2E model (using the StrathE2E() function)
#' it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new files.
#'
#' @param model R-list object generated by the read_model() function which defined the model configuration
#' @param nyears Number of years to run the model in each iteration (default=40)
#' @param n_iter Number of iterations of the model (default=500)
#' @param start_temperature Initial value of the simulated annealing temperature parameter (default=1). Suggested values in the range 0.0005 - 5. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood
#' @param cooling Rate at which the simulated annealing temperature declines with iterations (default=0.975). Suggested values in the range 0.9 - 0.985
#' @param toppredlock (TRUE or FALSE) Locks-down the uptake parameters of the birds pinnipeds and cetaceans as these are hard to fit alongside the other parameters (default=TRUE)
#' @param quiet (TRUE or FALSE) Suppress informational messages at the start of each iteration (default=TRUE)
#'
#' @return CSV files of fitted parameters and the histories of the proposed and accepted parameter values
#'
#' @seealso \code{\link{list_models}} , \code{\link{read_model}} , \code{\link{StrathE2E}} , \code{\link{annealing_find_harvest_ratio_mult}} , \code{\link{annealing_find_gear_activity_mult_ecosystem_target}} , \code{\link{annealing_find_gear_activity_mult_HR_target}}
#'
#' @importFrom stats runif
#'
#' @export
#'
#' @examples
#' #Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#' copy_model("North_Sea", "2003-2013",
#'            dest.path="C:/Users/username/Documents/Models")
#'
#' # Load the 2003-2013 version of the North Sea model from the user workspace:
#' model<-read_model(model.name="North_Sea",
#'                   model.variant="2003-2013",
#'                   model.ident="TEST",
#'                   user.path="C:/Users/username/Documents/Models")
#'
#' # Quick Demo of the annealing function in operation:
#' annealing_find_ecology_parameters(model, nyears=5, n_iter=10, start_temperature=0.5)
#' # More realistic configuration would be (WARNING - this will take about 26 hours to run) :
#' annealing_find_ecology_parameters(model, nyears=50, n_iter=1000, start_temperature=1)
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

annealing_find_ecology_parameters <- function(model, nyears=40, n_iter=500, start_temperature=1, cooling=0.975, toppredlock=TRUE, quiet=TRUE) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	setup				<- elt(model, "setup")
	read.only			<- elt(setup, "read.only")
	model.path			<- elt(setup, "model.path")
	resultsdir			<- elt(setup, "resultsdir")
	identifier			<- elt(setup, "model.ident")

	data				<- elt(model, "data")
	fitted.parameters		<- elt(data, "fitted.parameters")
	HRscale_vector_multiplier	<- elt(data, "fleet.model", "HRscale_vector_multiplier")

	if (read.only) {
		cat("Warning: cannot write fitted parameters back to the model input folders - model is read-only\n")
		cat("Warning: to fix this, make a copy of the model using copy_model() into your own workspace.\n")
		stop("Model is not writable!")
	}

	print(date())

#	toppredlock <- TRUE		# LOCK THE BIRD SEAL AND CETACEAN UPTAKE PARAMETERES OR NOT.....

	# Set the annealing parameters:
#	n_iter		<- 1000		# Maximum number of iterations of the parameter randomisation process
#n_iter=5 # ZZ temp
#	temperature	<- 0.00005	# values 0.00005 to 5. Higher values make the annealing process accept more bad jumps in th early stages 
#	cooling		<- 0.975	# values < 1. Suggest do not change this

	if (length(which(HRscale_vector_multiplier<1 | HRscale_vector_multiplier>1)) > 0) {
		print("**************************************************************************")
		print("WARNING - one or more baseline Harvest Ratio scaling values differs from 1")
		print("**************************************************************************")
	}

	datastore <- c(
		fitted.parameters,
		annual_obj = 1e-60
	)

	perturbed       <- as.data.frame(datastore)
	parhistory	<- as.data.frame(datastore)		# these are all data frames
	proposalhistory	<- parhistory

	n_acceptances	<- 0					# Counter for the number of parameter acceptances which have occurred

	temperature<-start_temperature

	#ITERATE THE ANNEALING PROCESS.....
	#
	for (kkk in 1:n_iter){

		annealing.parms <- read_SD_ecology(model.path)	# read every loop so control is available during a run
		if (kkk>1) {

		# Jiggle the parameter values and load them into the vector which is passed to the model function
		perturbed <- perturb_parameters(datastore, annealing.parms, toppredlock)

		# perturbed parameters will be built into model.parameters for run:
		model$data$fitted.parameters <- perturbed[1:(length(perturbed)-1)]

		}

		# Build and run the model:
		build <- build_model(model, nyears)
		fit <- StrathE2E_fit(model, build, quiet)
		annual_obj <- elt(fit, "annual_obj")

		perturbed$annual_obj <- annual_obj

		temperature<-temperature*cooling

		if (kkk==1) lastbest <- annual_obj

		#Now the Metropolis algorithm.....
		#
		lik_ratio<-exp(((log(annual_obj)) - log(lastbest))/temperature)
		rand<-runif(1,0,1)
		new_accepted<-" Accepted: NO"
		if(lik_ratio>rand){
			n_acceptances<-n_acceptances+1
			datastore <- perturbed
			lastbest <- annual_obj
			new_accepted <- " Accepted: YES"
		}

		#--------------------------------

		if(kkk==1) proposalhistory<-perturbed # first run through perturbed = the unaltered initial parameter set
		if(kkk>1) proposalhistory<-rbind(proposalhistory, perturbed)
		filename <- csvname(resultsdir, "annealing_par-proposalhistory", identifier)
		writecsv(proposalhistory, filename, row.names=FALSE)

		if(kkk==1) parhistory<-perturbed  # first run through perturbed = the unaltered initial parameter set
		if(kkk>1) parhistory<-rbind(parhistory, datastore)
		filename <- csvname(resultsdir, "annealing_par-acceptedhistory", identifier)
		writecsv(parhistory, filename, row.names=FALSE)

		print(paste("Itn: ",kkk," ","Proposal: ", annual_obj,"   ",new_accepted,sep=""))

		#Plot or update the time series of proposal and acepted likelihoods so far....
		if(kkk>1){
		par(mfrow=c(1,1))
		axmin <- elt(annealing.parms, "axmin")
		axmax <- elt(annealing.parms, "axmax")
		plot(seq(1,nrow(proposalhistory)),proposalhistory$annual_obj,ylim=c(axmin,axmax),xlim=c(1,kkk),xlab="Iterations",ylab="Target data likelihood",type="l",col="grey")
		points(seq(1,nrow(parhistory)),parhistory$annual_obj,type="l",col="black",lwd=3)
                legend("topleft",c("accepted","proposed"),bg="transparent",col=c("black","grey"),lty=c(1,1),lwd=c(3,1),pt.cex=c(1,1))
		}
	}

	#At the conclusion of the whole process, extract the final row of the parhistory into the format of model fitted parameter files and save to disc
	write_fitted_parameters(model, parhistory)
}

