#
# box_and_whisker_monthly_plots_with_credible_intervals.R
#
#' Box-and-whisker plots of observed monthly data on the state of the ecosystem with equivalent properties derived from a Monte Carlo StrathE2E run
#'
#' Creates a multi-panel plot comparing annual cycles of monthly averaged observational data on nutrinet and plankton concentrations together with the distribution of credible values of equivalent data derived from the final year of a model run generated by the Monte_Carlo_StrathE2E() function
#'
#' For details of how the distribution of credible output values from StrathE2E are calculated see ?Monte_Carlo_StrathE2E.
#'
#' The function plots a multi-panel page of box-and-whisker plots showing the medians and variability ranges (quartiles as box-and-whisker) of observational data on nitrate, ammonia, chlorophyll, omnivorous and carnivorous zooplankton, and benthos larvae concentrations
#' (shown in black), alongside comparable box-and-whisker plots (shown in red) of equivalent measures derived from the final years of an ensemble of model runs generted by a Monte Carlo methodology (Monte_Carlo_StrathE2E() function)
#'
#' The observational data to be plotted are loaded from the folder Modelname/Variantname/Target_data/monthly_target_data-*.csv as part of a read_model() function call and are built into the R-list object generated by read_model().
#'
#' Optionally the function can read an example data set for one of the two North Sea model variants supplied with the package.
#'
#' The corresponding measures derived from the final year of a model run generated by the Monte_Carlo_StrathE2E() function call are located in /results/Modelname/Variantname/CredInt_processed_monthly_mass-*.csv, and in the R-list object generated by Monte_Carlo_StrathE2E()
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param use.example (TRUE or FALSE) Option to use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE)
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{Monte_Carlo_StrathE2E}}, \code{\link{box_and_whisker_monthly_plots}}, \code{\link{box_and_whisker_annual_plots_with_credible_intervals}} 
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "1970-1999")
#'
#' # Either run the Monte_Carlo_StrathE2E() to generate some results, or use the internal example data
#'
#' # Plot the final year comparison of observed and modelled ecosystem state on screen:
#' box_and_whisker_monthly_plots_with_credible_intervals(model, use.example=TRUE)
#'
#' # Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	box_and_whisker_monthly_plots_with_credible_intervals(model, use.example=TRUE)
#' dev.off()
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

box_and_whisker_monthly_plots_with_credible_intervals <- function(model, use.example=FALSE) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	resultsdir	<- elt(model, "setup", "resultsdir")
	model.path	<- elt(model, "setup", "model.path")
	model.ident	<- elt(model, "setup", "model.ident")
	model.name 		<- elt(model, "setup", "model.name")
	model.variant 		<- elt(model, "setup", "model.variant")

	#Read the observed data file
	#Format expected = 7 columns
	#Month	Variable	median	lower_centile	upper_centile	Units	low_cent_value	upp_cent_value	Comments
	#The variable names expected are:
	#surface_nitrate
	#deep_nitrate
	#surface_ammonia
	#deep_ammonia
	#surface_chlorophyll
	#omniv_zooplankton
	#carniv_zooplankton
	#larvae_susp_dep_benthos
	#larvae_carn_scav_benthos
	obstargetdataset	<- get.model.file(model.path, TARGET_DATA_DIR, file.pattern=MONTHLY_TARGET_DATA)


corefilename<-"CredInt_processed_monthly_mass"

if(use.example==TRUE){
	cfilename<-paste(CREDINT_DIR,"/",corefilename,sep="")
	credfile <- examplecsvname(model.name, model.variant, cfilename)
	cat(paste("Reading internally stored example results for the ",model.name," ",model.variant," model","\n",sep=""))
}

if(use.example==FALSE){
	credpath	<- makepath(resultsdir, CREDINT_DIR)
	credfile	<- csvname(credpath, corefilename, model.ident)
	cat("Reading credible interval processed data from '", credfile, "'\n", sep="")
}

	if (! file.exists(credfile)) {
		cat("Error: cannot find credible interval output file: ", credfile, "\n")
		stop("Please run the Monte-Carlo function!\n")
	}


	credintervaldata	<- readcsv(credfile, row.names=1)	# first column is row names

	#Make an array to hold the synthesised observational monthly data
	ntargobs<-100
	monthtarget<-array(0,dim=c(ntargobs,12,10))	#dimensions observations,month, parameter, 
	monlab<-c("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")

	installtargdata<-function(qr,obspar,monthtarget){
		#Make sure that the selected data are in the correct order
		data<-data.frame("Month"=monlab,"median"=rep(NA,12),"17centile"=rep(NA,12),"83centile"=rep(NA,12))
		for(mmm in 1:12){
			data[mmm,2:4]<-seldata[(which(seldata$Month==data$Month[mmm])),3:5]
		}
		for(mmm in 1:12){
			sdpos<-(data[mmm,4]-data[mmm,2])/(qr/2)
			sdneg<-(data[mmm,2]-data[mmm,3])/(qr/2)
			for(kkk in 1:ntargobs){
				rand<-rnorm(1,0,1)
				if(rand<0) dev<-rand*sdneg
				if(rand>=0) dev<-rand*sdpos
				monthtarget[kkk,mmm,obspar]<-data[mmm,2]+dev
			}
		}
		return(monthtarget)
	}

#~~~~~~~~~~~~~
#Read in the monthly surface nitrate target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="surface_nitrate")
obspar<-1
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly deep nitrate target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="deep_nitrate")
obspar<-2
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly surface ammonia target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="surface_ammonia")
obspar<-3
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly deep ammonia target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="deep_ammonia")
obspar<-4
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly surface chlorophyll target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="surface_chlorophyll")
obspar<-5
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly omniv zooplankton target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="omniv_zooplankton")
obspar<-6
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly carniv zooplankton target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="carniv_zooplankton")
obspar<-7
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly larvae of susp/dep feeding benthos target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="larvae_susp_dep_benthos")
obspar<-8
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)


#~~~~~~~~~~~~~
#Read in the monthly larvae of carn/scav feeding benthos target data file
#qr=number of sd in the observed quantile range
#obspar=position of the data in the data array
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="larvae_carn_scav_benthos")
obspar<-9
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)



#~~~~~~~~~~~~~
#COMBINE THE TWO TYPES OF BENTHIC LARVAE
#Read in the monthly susp benlar target data file
seldata<-subset(obstargetdataset,obstargetdataset$Variable=="larvae_susp_dep_benthos")
#Read in the monthly carn benlar target data file
seldata2<-subset(obstargetdataset,obstargetdataset$Variable=="larvae_carn_scav_benthos")
seldata[,3]<-seldata[,3]+seldata2[,3]
seldata[,4]<-seldata[,4]+seldata2[,4]
seldata[,5]<-seldata[,5]+seldata2[,5]
obspar<-10
qr<-3.5
if(seldata[1,7]==17 & seldata[1,8]==83) qr<-3.5
if(seldata[1,7]==5 & seldata[1,8]==95) qr<-7
monthtarget<-installtargdata(qr,obspar,monthtarget)



#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Generate the list objects needed by the bxp plotting function
	for(iii in 1:9){

		credrows<- seq(   ((iii-1)*(5+1))+2,((iii-1)*(5+1))+(5+1)   )

		modeldata2plot<-(credintervaldata[credrows,1])
		for(jj in 2:12) { modeldata2plot<-c(modeldata2plot,(credintervaldata[credrows,jj]))}

		if(iii==5) modeldata2plot<-modeldata2plot*4/2   # chnages assumed C/chl ratio from 40 to 20 <<<<<<<<<<<<<

		array2plot<- array(dim=c(5,12),modeldata2plot)
		bxpdata<-list(stats=array2plot,n=rep(100,12),conf=NULL,out=numeric(length=0))
		# bxp(bxpdata,boxwex=0.3,at=seq(1,12)+0.35,xlim=c(0,13),ylim=c(0,max(modeldata2plot)*1.1))
		if(iii==1) bxpdata1<-bxpdata
		if(iii==2) bxpdata2<-bxpdata
		if(iii==3) bxpdata3<-bxpdata
		if(iii==4) bxpdata4<-bxpdata
		if(iii==5) bxpdata5<-bxpdata
		if(iii==6) bxpdata6<-bxpdata
		if(iii==7) bxpdata7<-bxpdata
		if(iii==8) bxpdata8<-bxpdata
		if(iii==9) bxpdata9<-bxpdata
	}

	bxpdata10<-list(stats=(bxpdata9$stats+bxpdata8$stats),n=rep(100,12),conf=NULL,out=numeric(length=0))
	#Combines the two types of benthic larvae

	#~~~~~~~~~~~~~~~
	plotdata<-function(obspar){
		obsplot<-as.data.frame(monthtarget[,,obspar])
		names(obsplot)<-monlab
		#modplot<-as.data.frame(MODmonthtarget[,,obspar])

		if(obspar==1){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,30),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Surf.nitrate mM.m"^{-3},sep="")))
			bxp(bxpdata1,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==2){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,30),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Deep nitrate mM.m"^{-3},sep="")))
			bxp(bxpdata2,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==3){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,10),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Surf.ammonia mM.m"^{-3},sep="")))
			bxp(bxpdata3,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==4){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,10),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Deep ammonia mM.m"^{-3},sep="")))
			bxp(bxpdata4,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==5){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,8),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Chlorophyll mg.m"^{-3},sep="")))
			bxp(bxpdata5,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==6){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,3),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Omniv.zoo. mMN.m"^{-3},sep="")))
			bxp(bxpdata6,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==7){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,0.5),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Carniv.zoo. mMN.m"^{-3},sep="")))
			bxp(bxpdata7,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==8){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,3),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Larv.s/d.benth. mMN.m"^{-3},sep="")))
			bxp(bxpdata8,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==9){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,0.6),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Larv.c/s.benth. mMN.m"^{-3},sep="")))
			bxp(bxpdata9,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		if(obspar==10){
			boxplot(obsplot,range=0,boxwex=0.25,ylim=c(0,3.0),las=1,par(mar=c(2.5,5,1,1)),cex.axis=1.3,cex.lab=1.3,ylab=expression(paste("Larv.benth. mMN.m"^{-3},sep="")))
			bxp(bxpdata10,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
		}

		#boxplot(modplot,add=TRUE,range=0,boxwex=0.25,at=1:12+0.3,par(lty=1),yaxt="n",xaxt="n")
	}
	#~~~~~~~~~~~~~~~

	#jpeg(paste(credpath,"monthly_model_vs_data.jpg",sep=""),width=500,height=500)
	#png(paste(credpath,"monthly_model_vs_data.png",sep=""),width=500,height=500)
	#pdf(paste(credpath,"monthly_model_vs_data.pdf",sep=""),width=6,height=9)
	#postscript(paste(credpath,"monthly_model_vs_data.ps",sep=""),width=6,height=6,horizontal=FALSE)

	par(mfrow=c(4,2))

	plotdata(1) # surface nitrate

	plotdata(2) # deep nitrate

	plotdata(3) # surface ammonia

	plotdata(4) # deep ammonia

	plotdata(5) # chlorophyll

	plotdata(6) # omnivorous zooplankton

	plotdata(7) # carnivorous zooplankton

	#plotdata(8) # susp/dep feeding benthos larvae (not plotted)

	#plotdata(9) # carn/scav feeding benthos larvae (not plotted)

	plotdata(10) # combined susp/dep and carn/scav feeding benthos larvae

	#dev.off()

}

