#
# concatenate_raw_sensitivity_analysis_results.R
#
#' Combine two or more sets of raw output data from sensitivity analysis runs performed on separate machine/processors
#'
#' The function Sensitivity_analysis_StrathE2E() is extremely time consuming so it makes sense to share the 
#' load across multiple processors and combine the results afterwards. This function concatenates the raw outputs from two separate runs
#' of Sensitivity_analysis_StrathE2E() into single a single file.
#'
#' Sensitivity_analysis_StrathE2E() generates two output files per run - OAT_results-*.csv, and OAT_parameter_values-*.csv, where * is
#' an identifying text string set by an argument of the read_model() function, This concatenation function combines both of these types of files.
#'
#' The files to be combined must be transferred into the same folder, and this is where the new combined file will be placed.
#' The path to locate the files is set in the read_model() function. If not specified it is assumed that the files are located in the
#' default /results/Modelname/Variantname folder in the current user workspace. 
#'
#' The list of files to be combined (any number > 1) is defined by a vector of their "model.ident" identifiers ("ident.list" argument). 
#' The first-named model.ident inthe vector should correspond to a run of the Sensitivity_analysis_StrathE2E() function with the argument coldstart=TRUE. Thus forces the
#' first trajectory of sensitivity test to be performed on the maximum-likelihood parameter set loaded with the read_model() function. Thus is important
#' for the post-processing stage of the analysis which needs to be performed on the combined results.
#' 
#' When combining the files, the function creates a seamless sequence of trajectory identifiers through the combined data, beginning from 1 for the first trajectory 
#' of the first set.
#'
#' If for any reason there are more than two separate sets of Sensitivity_analysis_StrathE2E() run results to be combined, then 
#' post-processing of the final combined file can be delayed with the 'postprocess' argument until the last concatenation when all the data have been gathered
#' together - using the function process_sensitivity_analysis_results_offline().
#'
#' A separate function plot_sensitivity_analysis_results() produces a graphical representation of the post-processed results of the combined data.
#'
#' @references Morris, M.D. (1991). Factorial sampling plans for preliminary computational experiments. Technometrics, 33, 161-174.
#'
#' @param model Model object for the raw data to be combined
#' @param ident.list A vector of text variables corresponding to the "model.ident" identifiers for each of the files to concatenated (list must be length 2 or greater)
#' @param combined.ident Identifier for the combined set of data, equivalent to the model.ident argument in read_model(). Default ="COMBINED"
#' @param postprocess (TRUE or FALSE, default=TRUE) Process the results through to a final sorted list of parameter sensitivies for plotting, or just produce the combined raw results. The reason for NOT processing would be if there are further run results stil to be combined with the set produced by this function.
#'
#' @return CSV files of parameter vectors for each model run, likelihoods and Elementary Effects for each run. If the argument postprocess=TRUE then also a dtaframe and CSV file of the parameter list sorted by EE_mean, from the combination of multiple input data sets
#'
#' @seealso \code{\link{read_model}} , \code{\link{Sensitivity_analysis_StrathE2E}} , \code{\link{process_sensitivity_analysis_results_offline}} , \code{\link{plot_sensitivity_analysis_results}}
#'
#' @export
#'
#' @examples
#' # The example here assumes that three sets of raw results data have been previously
#' # generated by runs of the Sensitivity_analysis_StrathE2E() function, with the model.ident values
#' # assigned as "RUN1", "RUN2" and "RUN3", and that these have been gathered together in the same
#' # /results/Modelname/Variantname folder.
#'
#' # Load the model name and version for the data to be combined :
#' # E.g. ... model <- read_model("North_Sea", "2003-2013")
#' sens_results <- concatenate_raw_sensitivity_analysis_results(model, ident.list=c("RUN1","RUN2","RUN3"), combined.ident="TESTCOMB", postprocess=TRUE)
#' head(sens_results)
#

concatenate_raw_sensitivity_analysis_results <- function(model, ident.list, combined.ident="COMBINED", postprocess=TRUE) {

	resultsdir <- elt(model, "setup", "resultsdir")

        Nfiles<-length(ident.list)

	if(Nfiles<2) {
		print("Less than 2 input files in the ident.list so function terminated")
		stop
	}

	for(qq in 1:Nfiles) {

	sensfile<- csvname(resultsdir, "OAT_results", ident.list[qq])
	parmfile<- csvname(resultsdir, "OAT_parameter_values", ident.list[qq])
	print(paste("Reading input data for model.ident = ",ident.list[qq],sep=""))
	results_df_out<- readcsv(sensfile)
	OAT_parmvalues<- readcsv(parmfile)

	if(qq==1){
		results_df_out_comb <- results_df_out
		OAT_parmvalues_comb<-OAT_parmvalues
		NPARMST<-(nrow(results_df_out[which(results_df_out$trajectoryid==1),]))-1
		NTRAJT<-(nrow(results_df_out))/(NPARMST+1)
	}
		

	if(qq>1){
		NPARMS<-(nrow(results_df_out[which(results_df_out$trajectoryid==1),]))-1
			if((NPARMST==NPARMS)==FALSE){
				print("Error: mismatch in parameter count between input files")
				stop
			}
		NTRAJ<-(nrow(results_df_out))/(NPARMS+1)
	
		results_df_out$trajectoryid <- results_df_out$trajectoryid + NTRAJT
		NTRAJT<-NTRAJT+NTRAJ
		results_df_out_comb <- rbind(results_df_out_comb, results_df_out)
		OAT_parmvalues_comb <- rbind(OAT_parmvalues_comb, OAT_parmvalues)
	}

	}

	filename <- csvname(resultsdir, "OAT_results", combined.ident)
	writecsv(results_df_out_comb, filename, row.names=FALSE)

	filename <- csvname(resultsdir, "OAT_parameter_values", combined.ident)
	writecsv(OAT_parmvalues_comb, filename, row.names=FALSE)


	#PROCESS THE FINAL RESULTS TO GET THE SORTED PARAMETER SENSITIVITY FILE IF REQUIRED
	Sorted_SENS_results<-NULL
	if(postprocess==TRUE){
	        model$setup$model.ident <- combined.ident
		Sorted_SENS_results<-process_sensitivity_analysis_results(model, results_df_out_comb)
        }

	return(Sorted_SENS_results)

}

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~