#
# plot_final_year.R
#
#' Multi-panel time series plots of state variable data for the final year of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series plots of daily state variable values aggregated over the inshore and offshore zones of the domain and over sediment classes, for the final year of a model run.
#' Currently the masses of macrophytes, corpses and discards are not included in the plots due to space constraints.
#'
#' Units of the plotted varaiables are mass concentrations, scaled to seas surface area or layer volume, ie. mMN/m2 or mMN/m3
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_full_length_timeseries}} , \code{\link{plot_final_year_HTL_inshore_vs_offshore}} , \code{\link{plot_final_year_LTL_inshore_vs_offshore}} , \code{\link{plot_sediment_habitats_final_year}}
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the final year of outputs
#' plot_final_year(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_final_year(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_final_year(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the final year output
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_final_year(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_final_year(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

plot_final_year <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	build		<- elt(results, "build")
	run		<- elt(build, "run")
	nyears		<- elt(run, "nyears")
	ndays		<- elt(run, "ndays")

	data		<- elt(model, "data")
	physical.parms	<- elt(data, "physical.parameters")
	si_depth	<- elt(physical.parms, "si_depth")
	so_depth	<- elt(physical.parms, "so_depth")
	d_depth		<- elt(physical.parms, "d_depth")
	x_shallowprop	<- elt(physical.parms, "x_shallowprop")

	output		<- elt(results, "output")
	detritus_d	<- elt(output, "detritus_d")
	nitrate_d	<- elt(output, "nitrate_d")
	ammonia_d	<- elt(output, "ammonia_d")
	phyt_d		<- elt(output, "phyt_d")

	aggregates	<- elt(results, "aggregates")
	x_poros		<- elt(aggregates, "x_poros")
	x_depth		<- elt(aggregates, "x_depth")
	s_detritus	<- elt(aggregates, "s_detritus")
	x_detritus	<- elt(aggregates, "x_detritus")
	s_nitrate	<- elt(aggregates, "s_nitrate")
	x_nitrate	<- elt(aggregates, "x_nitrate")
	s_ammonia	<- elt(aggregates, "s_ammonia")
	x_ammonia	<- elt(aggregates, "x_ammonia")
	s_phyt		<- elt(aggregates, "s_phyt")
	herb		<- elt(aggregates, "herb")
	carn		<- elt(aggregates, "carn")
	benths		<- elt(aggregates, "benths")
	benthc		<- elt(aggregates, "benthc")
	benthslar	<- elt(aggregates, "benthslar")
	benthclar	<- elt(aggregates, "benthclar")
	fishp		<- elt(aggregates, "fishp")
	fishplar	<- elt(aggregates, "fishplar")
	fishm		<- elt(aggregates, "fishm")
	fishd		<- elt(aggregates, "fishd")
	fishdlar	<- elt(aggregates, "fishdlar")
	bird		<- elt(aggregates, "bird")
	seal		<- elt(aggregates, "seal")
	ceta		<- elt(aggregates, "ceta")

	xvolume_si<-si_depth*x_shallowprop
	xvolume_so<-so_depth*(1-x_shallowprop)
	xd_volume<-d_depth*(1-x_shallowprop)
	xs_volume <- xvolume_si + xvolume_so

par(mfrow=c(3,3))

seddetscalef<-100
l1<-s_detritus[((nyears-1)*360+1):ndays]/xs_volume
l2<-detritus_d[((nyears-1)*360+1):ndays]/xd_volume
l3<-seddetscalef*100*(((x_detritus[((nyears-1)*360+1):ndays])*14)/1000)/(x_depth*(((1-x_poros)*(2650*1000))))
#This converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)
#Then scale by seddetscalef to get on same axes as water detritus mMN/m3 water
fyplot3("Detritus","Nitrogen/m3","Surface","Deep","DrySed%N (*100)",l1,l2,l3)

l1<-s_nitrate[((nyears-1)*360+1):ndays]/xs_volume
l2<-nitrate_d[((nyears-1)*360+1):ndays]/xd_volume
l3<-x_nitrate[((nyears-1)*360+1):ndays]/(x_depth*x_poros)
#This converts the sediment nitrate into units of N /m3 in the pore water)
fyplot3("Nitrate","Nitrogen/m3","Surface","Deep","Pore water",l1,l2,l3)

l1<-s_ammonia[((nyears-1)*360+1):ndays]/xs_volume
l2<-ammonia_d[((nyears-1)*360+1):ndays]/xd_volume
l3<-(x_ammonia[((nyears-1)*360+1):ndays]/(x_depth*x_poros))/10
#This converts the sediment nitrate into units of N /m3 in the pore water)
fyplot3("Ammonia","Nitrogen/m3","Surface","Deep","Pore water  (/10)",l1,l2,l3)

l1<-s_phyt[((nyears-1)*360+1):ndays]/xs_volume
l2<-phyt_d[((nyears-1)*360+1):ndays]/xd_volume
fyplot2("Phytoplankton","Nitrogen/m3","Surface","Deep",l1,l2)

l1<-herb[((nyears-1)*360+1):ndays]/(xs_volume+xd_volume)
l2<-carn[((nyears-1)*360+1):ndays]*10/(xs_volume+xd_volume)
fyplot2("Zooplankton","Nitrogen/m3","Omnivores","Carnivores (*10)",l1,l2)

#l1<-benths[((nyears-1)*360+1):ndays]
#l2<-benthc[((nyears-1)*360+1):ndays]*10
#fyplot2("Benthos","Nitrogen/m2","Susp/dep","Carnivores (*10)",l1,l2)

l1<-benths[((nyears-1)*360+1):ndays]
l2<-benthc[((nyears-1)*360+1):ndays]*10
l3<-benthslar[((nyears-1)*360+1):ndays]*10
l4<-benthclar[((nyears-1)*360+1):ndays]*100
fyplot4("Benthos","Nitrogen/m2","Susp/dep","Carniv (*10)","Susp/deplar (*10)","Carnlar (*100)",l1,l2,l3,l4)

l1<-fishp[((nyears-1)*360+1):ndays]
l2<-fishplar[((nyears-1)*360+1):ndays]
l3<-fishm[((nyears-1)*360+1):ndays]
fyplot3("Plank/Mig. fish","Nitrogen/m2","Adults","Larvae","Migratory",l1,l2,l3)

l1<-fishd[((nyears-1)*360+1):ndays]
l2<-fishdlar[((nyears-1)*360+1):ndays]
fyplot2("Demersal fish","Nitrogen/m2","Adults","Larvae",l1,l2)

l1<-bird[((nyears-1)*360+1):ndays]
l2<-seal[((nyears-1)*360+1):ndays]
l3<-ceta[((nyears-1)*360+1):ndays]
#fyplot1("Birds & mammals","Nitrogen/m2",l1)
fyplot3("Birds & mammals","Nitrogen/m2","Birds","Pinnipeds","Cetaceans",l1,l2,l3)


}

