#
# plot_final_year_HTL_inshore_vs_offshore.R
#
#' Plots of inshore and offshore high trophic level state variables for the final year of a model run
#'
#' Multi-panel time series plots of inshore and offshore high trophic level state variables for the final year of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series plots of daily values of carnivore/scaavenge feeding benthos (and their lavae), fish (and their larvae), birds, pinnipeds and cetaceans for the inshore and offshore zones, for the final year of a model run.
#'
#' Units of the plotted varaiables are mass concentrations, scaled to seas surface area or layer volume, ie. mMN/m2 or mMN/m3
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_final_year}} , \code{\link{plot_final_year_LTL_inshore_vs_offshore}} , \code{\link{plot_sediment_habitats_final_year}}
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the final year of high trophic level outputs
#' plot_final_year_HTL_inshore_vs_offshore(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_final_year_HTL_inshore_vs_offshore(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_final_year_HTL_inshore_vs_offshore(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the final year high trophic level output
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_final_year_HTL_inshore_vs_offshore(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_final_year_HTL_inshore_vs_offshore(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

plot_final_year_HTL_inshore_vs_offshore <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	build		<- elt(results, "build")
	run		<- elt(build, "run")
	nyears		<- elt(run, "nyears")
	ndays		<- elt(run, "ndays")

	data		<- elt(model, "data")
	physical.parms	<- elt(data, "physical.parameters")
	si_depth	<- elt(physical.parms, "si_depth")
	so_depth	<- elt(physical.parms, "so_depth")
	d_depth		<- elt(physical.parms, "d_depth")
	x_shallowprop	<- elt(physical.parms, "x_shallowprop")

	output		<- elt(results, "output")
	benthc_o	<- elt(output, "benthc_o")
	benthc_i	<- elt(output, "benthc_i")
	benthclar_o	<- elt(output, "benthclar_o")
	benthclar_i	<- elt(output, "benthclar_i")
	fishp_o		<- elt(output, "fishp_o")
	fishp_i		<- elt(output, "fishp_i")
	fishplar_o	<- elt(output, "fishplar_o")
	fishplar_i	<- elt(output, "fishplar_i")
	fishd_o		<- elt(output, "fishd_o")
	fishd_i		<- elt(output, "fishd_i")
	fishdlar_o	<- elt(output, "fishdlar_o")
	fishdlar_i	<- elt(output, "fishdlar_i")
	fishm_o		<- elt(output, "fishm_o")
	fishm_i		<- elt(output, "fishm_i")
	bird_o		<- elt(output, "bird_o")
	bird_i		<- elt(output, "bird_i")
	ceta_o		<- elt(output, "ceta_o")
	ceta_i		<- elt(output, "ceta_i")
	seal_o		<- elt(output, "seal_o")
	seal_i		<- elt(output, "seal_i")

	aggregates	<- elt(results, "aggregates")
	x_poros		<- elt(aggregates, "x_poros")
	x_depth		<- elt(aggregates, "x_depth")

	xvolume_si<-si_depth*x_shallowprop
	xvolume_so<-so_depth*(1-x_shallowprop)
	xd_volume<-d_depth*(1-x_shallowprop)

	#Plot the final year of output

	par(mfrow=c(3,3))

	l1<-benthc_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-benthc_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Benthos c/s","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Benthos c/s","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-benthclar_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-benthclar_i[((nyears-1)*360+1):ndays]/xvolume_si
#	tsplot22("Benthos c/s larvae","Nitrogen/m3","Offshore","Inshore",l1,l2)
fyplot2("Benthos c/s larvae","Nitrogen/m3","Offshore","Inshore",l1,l2)

	l1<-fishp_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishp_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Planktiv. fish","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Planktiv. fish","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-fishplar_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishplar_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Planktiv. fish larvae","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Planktiv. fish larvae","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-fishd_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishd_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Demersal fish","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Demersal fish","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-fishdlar_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishdlar_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Demersal fish larvae","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Demersal fish larvae","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-fishm_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-fishm_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Migratory fish","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Migratory fish","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-bird_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-bird_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot22("Birds","Nitrogen/m2","Offshore","Inshore",l1,l2)
fyplot2("Birds","Nitrogen/m2","Offshore","Inshore",l1,l2)

	l1<-ceta_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-ceta_i[((nyears-1)*360+1):ndays]/x_shallowprop
	l3<-seal_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l4<-seal_i[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot44("Mammals","Nitrogen/m2","Ceatacean_o","Cetaceans_i", "Pinnipeds_o","Pinnipeds_i",l1,l2,l3,l4)
fyplot4("Mammals","Nitrogen/m2","Ceatacean_o","Cetaceans_i", "Pinnipeds_o","Pinnipeds_i",l1,l2,l3,l4)

}

