#
# plot_final_year_time_series_data_with_credible_intervals.R
#
#' Create multi-panel plots showing the credible intervals of state variable densities over the final year of a simulation generated by the Monte_Carlo_StrathE2E() function
#'
#' Each panel of the plot shows a time-series of the densities (per unit sea surface area or layer volume) of a
#' state variable over the final year of a run. The data displayed are the solution to the maximum likelihood model, together with 
#' centiles of the distribution of credible values. The distribution of credible values is derived from
#' a Monte Carlo simulation scheme performed by the Monte_Carlo_StrathE2E() function.
#'
#' Daily interval post-processed data from the Monte_Carlo_StrathE2E() function are stored in the file
#' /results/Modelname/Variantname/CredInt/CredInt_processed_daily_mass-*.csv, where * represents the model run identifier
#' (model.ident) text embedded in the R-list object created by the read_model() function.
#' The function described here loads these stored data. Depending on the value set for the 'selection' argument in the function,
#' different groups of variables are extracted for plotting in each multi-panel page.
#'
#' Optionally the function can read an example data set for one of the two North Sea model variants supplied with the package.
#'
#' In each plot the x-axis represents day of the year over an annual cycle. The y-axis represents mass density of a state variable
#' (mMN/m2 or mMN/m3). Results from the maximum likelihood model are shown by a red line. The median of the credible values distribution
#' is shown my a solid black line. A grey-shaded area indicates the 50% credible interval (spanning quartiles of the cumulative likelihood
#' of simulated values). Black dashed lines span the 99% credible interval.
#'
#' For details of how the distribution of credible output values from StrathE2E are calculated see ?Monte_Carlo_StrathE2E.
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param selection Text string from a list identifying the group of model output variables to be plotted. Select from: "NUT_PHYT", "SEDIMENT", "ZOOPLANKTON", "FISH", "BENTHOS", "PREDATORS", "CORP_DISC", "MACROPHYTE", Remember to include the phrase within "" quotes.
#' @param use.example (TRUE or FALSE) Option to use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE)
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{Monte_Carlo_StrathE2E}}, \code{\link{plot_final_year_migration_data_with_credible_intervals}} , \code{\link{plot_inshore_vs_offshore_anavmass_with_credible_intervals}} 
#'
#' @importFrom graphics polygon
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "1970-1999")
#'
#' # Either run the Monte_Carlo_StrathE2E() to generate some results, or use the internal example data
#'
#' # Plot final year time series from the internal example data on screen, each page to a new plot window:
#' plot_final_year_time_series_data_with_credible_intervals(model, "NUT_PHYT",use.example=TRUE)	   # plot nutrient and phytoplankton data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "SEDIMENT",use.example=TRUE)	   # plot sediment chemisty data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "ZOOPLANKTON",use.example=TRUE) # plot zooplankton data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "FISH",use.example=TRUE)	   # plot fish and fish larvae data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "BENTHOS",use.example=TRUE)	   # plot benthos and benthos larvae data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "PREDATORS",use.example=TRUE)   # plot bird, pinniped and cetacean data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "CORP_DISC",use.example=TRUE)   # plot corpse and discard data
#' dev.new()
#' plot_final_year_time_series_data_with_credible_intervals(model, "MACROPHYTE",use.example=TRUE)  # plot macrophyte data
#'
#' # To direct the graph output to a file rather than the screen, wrap the plot_final_year_data_with_credible_intervals() function call in a graphical device call:
#' # Since the plot pages contain different numbers of panels the recommended width:height ratios are as follows:
#' #	NUT_PHYT,	1.5  : 1
#' #	SEDIMENT,	0.67 : 1
#' #	ZOOPLANKTON,	1    : 1
#' #	FISH		2    : 1
#' #	BENTHOS		2    : 1
#' #	PREDATORS	2    : 1
#' #	CORP_DISC	1    : 1
#' #	MACROPHYTE	2    : 1
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=7,height=4.667)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "NUT_PHYT",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=4,height=6)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "SEDIMENT",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=4,height=4)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "ZOOPLANKTON",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=3)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "FISH",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=3)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "BENTHOS",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=3)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "PREDATORS",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=4,height=4)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "CORP_DISC",use.example=TRUE)
#' dev.off()
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=4,height=2)
#' 	plot_final_year_time_series_data_with_credible_intervals(model, "MACROPHYTE",use.example=TRUE)
#' dev.off()
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------


plot_final_year_time_series_data_with_credible_intervals <- function(model, selection, use.example=FALSE) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	resultsdir	<- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")
	model.name 	<- elt(model, "setup", "model.name")
	model.variant 	<- elt(model, "setup", "model.variant")

corefilename<-"CredInt_processed_daily_mass"

if(use.example==TRUE){
	cfilename<-paste(CREDINT_DIR,"/",corefilename,sep="")
	credfile <- examplecsvname(model.name, model.variant, cfilename)
	cat(paste("Reading internally stored example results for the ",model.name," ",model.variant," model","\n",sep=""))
}

if(use.example==FALSE){
	credpath	<- makepath(resultsdir, CREDINT_DIR)
	credfile	<- csvname(credpath, corefilename, model.ident)
	cat("Reading credible interval processed data from '", credfile, "'\n", sep="")
}

	Result_data_store <- readcsv(credfile, row.names=1)
	#showall("row.names()", row.names(Result_data_store))


# ---------------------------------------------------------------------------------

	# Function to find joint y axis maximum acorss several plots

	findyaxmax<-function(VtoGet){
	
	for(jjj in 1:(length(VtoGet))){
		if(jjj==1) alldata<-as.numeric(Result_data_store[VtoGet[jjj],])
		if(jjj>1)  alldata<-c(alldata,as.numeric(Result_data_store[VtoGet[jjj],]))
	}
	yaxmax<- max(alldata,na.rm=TRUE)
	yaxmax
	}

# ---------------------------------------------------------------------------------

	# Function to plot cred int data
	tsplotCI<-function(tspmain,axtitle,VtoGet,ymax){
	#In each set:
	#Row1 = maximum likelihood model result
	#Row2 = 0.005 centile
	#Row3 = lower quartile
	#Row4 = median
	#Row5 = upper quartile
	#Row6 = 0.995 centile
	#
		par(mar=c(3.5,3.8,2.1,0.7))

		cat("Plot", tspmain, "and", axtitle, "\n")

		yml<-as.numeric(Result_data_store[VtoGet[1],])
		yll<-as.numeric(Result_data_store[VtoGet[2],])
		ylq<-as.numeric(Result_data_store[VtoGet[3],])
		ymd<-as.numeric(Result_data_store[VtoGet[4],])
		yuq<-as.numeric(Result_data_store[VtoGet[5],])
		yul<-as.numeric(Result_data_store[VtoGet[6],])

#		showall("data", VtoGet)
#		showall("yml", yml)
#		showall("yll", yll)
#		showall("ylq", ylq)
#		showall("ymd", ymd)
#		showall("yuq", yuq)
#		showall("yul", yul)

		poly_y<-c(ylq,yuq[361:1],ylq[1])
		poly_x<-c(seq(0,360),seq(360,0,by=-1),0)

		if(ymax>0)  y_max<-ymax
		if(ymax==0) y_max<-1.1*max(yul)

		plot(seq(0,360),yul,ylim=c(0,y_max),type="l",lty="dashed",lwd=1,col="black",xaxt="n",yaxt="n",ann=FALSE,main=tspmain)
		polygon(poly_x,poly_y,col="grey",border="grey")
		lines(seq(0,360),yll,type="l",lty="dashed",lwd=1,col="black")
		lines(seq(0,360),ymd,type="l",lwd=2,col="black")
		lines(seq(0,360),yml,type="l",lwd=1,col="red")

		axis(side=1,at=c(0,60,120,180,240,300,360),las=1,cex.axis=0.9)
		axis(side=2,las=1,cex.axis=0.9)
		mtext("Days",cex=0.7,side=1,line=2)
		mtext(axtitle,cex=0.7,side=2,line=2.8)
		mtext(tspmain,cex=0.7,side=3,line=0.5)
	}

# ---------------------------------------------------------------------------------

	if (selection == "NUT_PHYT") {
		#WATER COLUMN NUTRIENTS DETRITUS AND PHYTOPLANKTON

		Surface_detritus_offshore_id		<-grep("Surface_detritus_offshore",row.names(Result_data_store))
		Surface_ammonia_offshore_id		<-grep("Surface_ammonia_offshore",row.names(Result_data_store))
		Surface_nitrate_offshore_id		<-grep("Surface_nitrate_offshore",row.names(Result_data_store))
		Surface_phytoplankton_offshore_id	<-grep("Surface_phytoplankton_offshore",row.names(Result_data_store))

		Deep_detritus_offshore_id		<-grep("Deep_detritus_offshore",row.names(Result_data_store))
		Deep_ammonia_offshore_id		<-grep("Deep_ammonia_offshore",row.names(Result_data_store))
		Deep_nitrate_offshore_id		<-grep("Deep_nitrate_offshore",row.names(Result_data_store))
		Deep_phytoplankton_offshore_id		<-grep("Deep_phytoplankton_offshore",row.names(Result_data_store))

		Surface_detritus_inshore_id		<-grep("Surface_detritus_inshore",row.names(Result_data_store))
		Surface_ammonia_inshore_id		<-grep("Surface_ammonia_inshore",row.names(Result_data_store))
		Surface_nitrate_inshore_id		<-grep("Surface_nitrate_inshore",row.names(Result_data_store))
		Surface_phytoplankton_inshore_id	<-grep("Surface_phytoplankton_inshore",row.names(Result_data_store))


		C1max<-1.2*findyaxmax(c(Surface_detritus_offshore_id,
				        Deep_detritus_offshore_id,
				        Surface_detritus_inshore_id))


		C2max<-1.2*findyaxmax(c(Surface_ammonia_offshore_id,
				        Deep_ammonia_offshore_id,
				        Surface_ammonia_inshore_id))


		C3max<-1.2*findyaxmax(c(Surface_nitrate_offshore_id,
				        Deep_nitrate_offshore_id,
				        Surface_nitrate_inshore_id))

		C4max<-1.2*findyaxmax(c(Surface_phytoplankton_offshore_id,
				        Deep_phytoplankton_offshore_id,
				        Surface_phytoplankton_inshore_id))

		par(mfrow=c(3,4))

		VtoGet<-Surface_detritus_offshore_id
		tspmain<-"Offshore zone upper layer"
		axtitle<-"Suspended bact & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Surface_ammonia_offshore_id
		tspmain<-"Offshore zone upper layer"
		axtitle<-"Ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Surface_nitrate_offshore_id
		tspmain<-"Offshore zone upper layer"
		axtitle<-"Nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Surface_phytoplankton_offshore_id
		tspmain<-"Offshore zone upper layer"
		axtitle<-"Phytoplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

		VtoGet<-Deep_detritus_offshore_id
		tspmain<-"Offshore zone lower layer"
		axtitle<-"Suspended bact & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Deep_ammonia_offshore_id
		tspmain<-"Offshore zone lower layer"
		axtitle<-"Ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Deep_nitrate_offshore_id
		tspmain<-"Offshore zone lower layer"
		axtitle<-"Nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Deep_phytoplankton_offshore_id
		tspmain<-"Offshore zone lower layer"
		axtitle<-"Phytoplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

		VtoGet<-Surface_detritus_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Suspended bact & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Surface_ammonia_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-" Ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Surface_nitrate_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Surface_phytoplankton_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Phytoplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)
	}

# ---------------------------------------------------------------------------------

	else if (selection == "SEDIMENT") {
		#SEABED DETRITUS AND NUTRIENTS

		Sediment_detritus_N_by_weight_offshore_mud_id	<-grep("Sediment_detritus_N%_by_weight_offshore_mud",row.names(Result_data_store))
		Sediment_detritus_N_by_weight_offshore_sand_id	<-grep("Sediment_detritus_N%_by_weight_offshore_sand",row.names(Result_data_store))
		Sediment_detritus_N_by_weight_inshore_mud_id	<-grep("Sediment_detritus_N%_by_weight_inshore_mud",row.names(Result_data_store))
		Sediment_detritus_N_by_weight_inshore_sand_id	<-grep("Sediment_detritus_N%_by_weight_inshore_sand",row.names(Result_data_store))

		Porewater_nitrate_offshore_mud_id		<-grep("Porewater_nitrate_offshore_mud",row.names(Result_data_store))
		Porewater_nitrate_offshore_sand_id		<-grep("Porewater_nitrate_offshore_sand",row.names(Result_data_store))
		Porewater_nitrate_inshore_mud_id		<-grep("Porewater_nitrate_inshore_mud",row.names(Result_data_store))
		Porewater_nitrate_inshore_sand_id		<-grep("Porewater_nitrate_inshore_sand",row.names(Result_data_store))

		Porewater_ammonia_offshore_mud_id		<-grep("Porewater_ammonia_offshore_mud",row.names(Result_data_store))
		Porewater_ammonia_offshore_sand_id		<-grep("Porewater_ammonia_offshore_sand",row.names(Result_data_store))
		Porewater_ammonia_inshore_mud_id		<-grep("Porewater_ammonia_inshore_mud",row.names(Result_data_store))
		Porewater_ammonia_inshore_sand_id		<-grep("Porewater_ammonia_inshore_sand",row.names(Result_data_store))


		C1max<-1.2*findyaxmax(c(Sediment_detritus_N_by_weight_offshore_mud_id,
				        Sediment_detritus_N_by_weight_offshore_sand_id,
				        Sediment_detritus_N_by_weight_inshore_mud_id,
				        Sediment_detritus_N_by_weight_inshore_sand_id))


		C2max<-1.2*findyaxmax(c(Porewater_nitrate_offshore_mud_id,
				        Porewater_nitrate_offshore_sand_id,
				        Porewater_nitrate_inshore_mud_id,
				        Porewater_nitrate_inshore_sand_id))


		C3max<-1.2*findyaxmax(c(Porewater_ammonia_offshore_mud_id,
				        Porewater_ammonia_offshore_sand_id,
				        Porewater_ammonia_inshore_mud_id,
				        Porewater_ammonia_inshore_sand_id))


		par(mfrow=c(4,3))

		VtoGet<-Sediment_detritus_N_by_weight_offshore_mud_id
		tspmain<-"Offshore mud"
		axtitle<-"Sediment bact. & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Porewater_nitrate_offshore_mud_id
		tspmain<-"Offshore mud"
		axtitle<-"Porewater nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Porewater_ammonia_offshore_mud_id
		tspmain<-"Offshore mud"
		axtitle<-"Porewater ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Sediment_detritus_N_by_weight_offshore_sand_id
		tspmain<-"Offshore sand"
		axtitle<-"Sediment bact. & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Porewater_nitrate_offshore_sand_id
		tspmain<-"Offshore sand"
		axtitle<-"Porewater nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Porewater_ammonia_offshore_sand_id
		tspmain<-"Offshore sand"
		axtitle<-"Porewater ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)


		VtoGet<-Sediment_detritus_N_by_weight_inshore_mud_id
		tspmain<-"Inshore mud"
		axtitle<-"Sediment bact. & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Porewater_nitrate_inshore_mud_id
		tspmain<-"Inshore mud"
		axtitle<-"Porewater nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Porewater_ammonia_inshore_mud_id
		tspmain<-"Inshore mud"
		axtitle<-"Porewater ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Sediment_detritus_N_by_weight_inshore_sand_id
		tspmain<-"Inshore sand"
		axtitle<-"Sediment bact. & detritus"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Porewater_nitrate_inshore_sand_id
		tspmain<-"Inshore sand"
		axtitle<-"Porewater nitrate"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Porewater_ammonia_inshore_sand_id
		tspmain<-"Inshore sand"
		axtitle<-"Porewater ammonia"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "ZOOPLANKTON") {
		#ZOOPLANKTON

		Omnivorous_zooplankton_offshore_id	<-grep("Omnivorous_zooplankton_offshore",row.names(Result_data_store))
		Omnivorous_zooplankton_inshore_id	<-grep("Omnivorous_zooplankton_inshore",row.names(Result_data_store))
		Carnivorous_zooplankton_offshore_id	<-grep("Carnivorous_zooplankton_offshore",row.names(Result_data_store))
		Carnivorous_zooplankton_inshore_id	<-grep("Carnivorous_zooplankton_inshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(c(Omnivorous_zooplankton_offshore_id,
				        Omnivorous_zooplankton_inshore_id))


		C2max<-1.2*findyaxmax(c(Carnivorous_zooplankton_offshore_id,
				        Carnivorous_zooplankton_inshore_id))


		par(mfrow=c(2,2))

		VtoGet<-Omnivorous_zooplankton_offshore_id
		tspmain<-"Offshore zone "
		axtitle<-"Omnivorous zooplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Carnivorous_zooplankton_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Carnivorous zooplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Omnivorous_zooplankton_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Omnivorous zooplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Carnivorous_zooplankton_inshore_id
		tspmain<-"Inshore zone "
		axtitle<-"Carnivorous zooplankton"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "FISH") {
		#FISH AND FISH LARVAE

		Planktivorous_fish_offshore_id		<-grep("Planktivorous_fish_offshore",row.names(Result_data_store))
		Planktivorous_fish_larvae_offshore_id	<-grep("Planktivorous_fish_larvae_offshore",row.names(Result_data_store))
		Demersal_fish_offshore_id		<-grep("Demersal_fish_offshore",row.names(Result_data_store))
		Demersal_fish_larvae_offshore_id	<-grep("Demersal_fish_larvae_offshore",row.names(Result_data_store))
		Planktivorous_fish_inshore_id		<-grep("Planktivorous_fish_inshore",row.names(Result_data_store))
		Planktivorous_fish_larvae_inshore_id	<-grep("Planktivorous_fish_larvae_inshore",row.names(Result_data_store))
		Demersal_fish_inshore_id		<-grep("Demersal_fish_inshore",row.names(Result_data_store))
		Demersal_fish_larvae_inshore_id		<-grep("Demersal_fish_larvae_inshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(c(Planktivorous_fish_offshore_id,
				        Planktivorous_fish_inshore_id))


		C2max<-1.2*findyaxmax(c(Planktivorous_fish_larvae_offshore_id,
				        Planktivorous_fish_larvae_inshore_id))


		C3max<-1.2*findyaxmax(c(Demersal_fish_offshore_id,
				        Demersal_fish_inshore_id))

		C4max<-1.2*findyaxmax(c(Demersal_fish_larvae_offshore_id,
				        Demersal_fish_larvae_inshore_id))

		par(mfrow=c(2,4))

		VtoGet<-Planktivorous_fish_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Planktivorous fish"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Planktivorous_fish_larvae_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Planktivorous fish larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Demersal_fish_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Demersal fish"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Demersal_fish_larvae_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Demersal fish larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

		VtoGet<-Planktivorous_fish_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Planktivorous fish"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Planktivorous_fish_larvae_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Planktivorous fish larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Demersal_fish_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Demersal fish"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Demersal_fish_larvae_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Demersal fish larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "BENTHOS") {
		#BENTHOS AND BENTHOS LARVAE

		Suspension_deposit_benthos_offshore_id		<-grep("Suspension_deposit_benthos_offshore",row.names(Result_data_store))
		Suspension_deposit_benthos_larvae_offshore_id	<-grep("Suspension_deposit_benthos_larvae_offshore",row.names(Result_data_store))
		Carnivore_scavenge_benthos_offshore_id		<-grep("Carnivore_scavenge_benthos_offshore",row.names(Result_data_store))
		Carnivore_scavenge_benthos_larvae_offshore_id	<-grep("Carnivore_scavenge_benthos_larvae_offshore",row.names(Result_data_store))
		Suspension_deposit_benthos_inshore_id		<-grep("Suspension_deposit_benthos_inshore",row.names(Result_data_store))
		Suspension_deposit_benthos_larvae_inshore_id	<-grep("Suspension_deposit_benthos_larvae_inshore",row.names(Result_data_store))
		Carnivore_scavenge_benthos_inshore_id		<-grep("Carnivore_scavenge_benthos_inshore",row.names(Result_data_store))
		Carnivore_scavenge_benthos_larvae_inshore_id	<-grep("Carnivore_scavenge_benthos_larvae_inshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(c(Suspension_deposit_benthos_offshore_id,
				        Suspension_deposit_benthos_inshore_id))


		C2max<-1.2*findyaxmax(c(Suspension_deposit_benthos_larvae_offshore_id,
				        Suspension_deposit_benthos_larvae_inshore_id))


		C3max<-1.2*findyaxmax(c(Carnivore_scavenge_benthos_offshore_id,
				        Carnivore_scavenge_benthos_inshore_id))

		C4max<-1.2*findyaxmax(c(Carnivore_scavenge_benthos_larvae_offshore_id,
				        Carnivore_scavenge_benthos_larvae_inshore_id))


		par(mfrow=c(2,4))

		VtoGet<-Suspension_deposit_benthos_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Susp./dep. benthos"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Suspension_deposit_benthos_larvae_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Susp./dep. benthos larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Carnivore_scavenge_benthos_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Carn./scav. benthos"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Carnivore_scavenge_benthos_larvae_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Carn./scav. benthos larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

		VtoGet<-Suspension_deposit_benthos_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Susp./dep. benthos"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Suspension_deposit_benthos_larvae_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Susp./dep. benthos larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Carnivore_scavenge_benthos_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Carn./scav. benthos"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Carnivore_scavenge_benthos_larvae_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Carn./scav. benthos larvae"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "PREDATORS") {
		#BIRDS/MAMMALS, MIGRATORY FISH

		Birds_offshore_id		<-grep("Birds_offshore",row.names(Result_data_store))
		Seals_offshore_id		<-grep("Pinnipeds_offshore",row.names(Result_data_store))
		Cetaceans_offshore_id		<-grep("Cetaceans_offshore",row.names(Result_data_store))
		Migratory_fish_inshore_id	<-grep("Migratory_fish_inshore",row.names(Result_data_store))
		Birds_inshore_id		<-grep("Birds_inshore",row.names(Result_data_store))
		Seals_inshore_id		<-grep("Pinnipeds_inshore",row.names(Result_data_store))
		Cetaceans_inshore_id		<-grep("Cetaceans_inshore",row.names(Result_data_store))
		Migratory_fish_offshore_id	<-grep("Migratory_fish_offshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(c(Birds_offshore_id,
				        Birds_inshore_id))


		C2max<-1.2*findyaxmax(c(Seals_offshore_id,
				        Seals_inshore_id))


		C3max<-1.2*findyaxmax(c(Cetaceans_offshore_id,
				        Cetaceans_inshore_id))

		C4max<-1.2*findyaxmax(c(Migratory_fish_inshore_id,
				        Migratory_fish_offshore_id))


		par(mfrow=c(2,4))

		VtoGet<-Birds_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Birds"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Seals_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Pinnipeds"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Cetaceans_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Cetaceans"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Migratory_fish_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Migratory fish"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

		VtoGet<-Birds_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Birds"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Seals_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Pinnipeds"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Cetaceans_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Cetaceans"
		tsplotCI(tspmain,axtitle,VtoGet,C3max)

		VtoGet<-Migratory_fish_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Migratory fish"
		tsplotCI(tspmain,axtitle,VtoGet,C4max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "CORP_DISC") {
		#CORPSES and DISCARDS

		Corpses_offshore_id	<-grep("Corpses_offshore",row.names(Result_data_store))
		Discards_offshore_id	<-grep("Discards_offshore",row.names(Result_data_store))
		Corpses_inshore_id	<-grep("Corpses_inshore",row.names(Result_data_store))
		Discards_inshore_id	<-grep("Discards_inshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(c(Corpses_offshore_id,
				        Corpses_inshore_id))


		C2max<-1.2*findyaxmax(c(Discards_offshore_id,
				        Discards_inshore_id))


		par(mfrow=c(2,2))

		VtoGet<-Corpses_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Seabed corpses"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Discards_offshore_id
		tspmain<-"Offshore zone"
		axtitle<-"Fishery discards"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

		VtoGet<-Corpses_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Seabed corpses"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Discards_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Fishery discards"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

	}

# ---------------------------------------------------------------------------------

	else if (selection == "MACROPHYTE") {
		#KELP and KELP DEBRIS

		Kelp_inshore_id		<-grep("Macrophytes_inshore",row.names(Result_data_store))
		Kelp_debris_inshore_id	<-grep("Macrophyte_debris_inshore",row.names(Result_data_store))

		C1max<-1.2*findyaxmax(Kelp_inshore_id)

		C2max<-1.2*findyaxmax(Kelp_debris_inshore_id)

		par(mfrow=c(1,2))

		VtoGet<-Kelp_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Macrophytes"
		tsplotCI(tspmain,axtitle,VtoGet,C1max)

		VtoGet<-Kelp_debris_inshore_id
		tspmain<-"Inshore zone"
		axtitle<-"Macrophyte debris"
		tsplotCI(tspmain,axtitle,VtoGet,C2max)

	}

# ---------------------------------------------------------------------------------

	else {
		stop("Error: unknown plot selection '", selection, "' !\n")
	}

}


