#
# plot_full_length_timeseries.R
#
#' Plot state variable data for the full duration of a model run
#'
#' Multi-panel time series plots of state variable data for the full duration of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series plots of daily state variable values aggregated over the inshore and offshore zones of the domain and over sediment classes, for the full duration of a model run.
#' Currently the masses of macrophytes, corpses and discards are not included in the plots due to space constraints.
#'
#' Be warned that if the run is more than about 10 years then the plot becomes extremely compressed and messy. It is not intended to
#' be of publication quality. The intention is to provide a quick-look diagnostic of trends in the state variables. This is useful to assess
#' whether the model is close to stationary state or not.
#'
#' Units of the plotted varaiables mMN ie. mass in the model domain without any scaling to zone-area or layer thickness
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_final_year}} , \code{\link{plot_final_year_HTL_inshore_vs_offshore}} , \code{\link{plot_final_year_LTL_inshore_vs_offshore}} , \code{\link{plot_sediment_habitats_final_year}}
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the time series of outputs
#' plot_full_length_timeseries(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_full_length_timeseries(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_full_length_timeseries(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the full length output
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_full_length_timeseries(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_full_length_timeseries(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

plot_full_length_timeseries <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	output		<- elt(results, "output")
	detritus_d	<- elt(output, "detritus_d")
	nitrate_d	<- elt(output, "nitrate_d")
	ammonia_d	<- elt(output, "ammonia_d")
	phyt_d		<- elt(output, "phyt_d")

	aggregates	<- elt(results, "aggregates")
	totalN		<- elt(aggregates, "totalN")
	s_detritus	<- elt(aggregates, "s_detritus")
	x_detritus	<- elt(aggregates, "x_detritus")
	s_nitrate	<- elt(aggregates, "s_nitrate")
	x_nitrate	<- elt(aggregates, "x_nitrate")
	s_ammonia	<- elt(aggregates, "s_ammonia")
	x_ammonia	<- elt(aggregates, "x_ammonia")
	s_phyt		<- elt(aggregates, "s_phyt")
	herb		<- elt(aggregates, "herb")
	carn		<- elt(aggregates, "carn")
	benths		<- elt(aggregates, "benths")
	benthslar	<- elt(aggregates, "benthslar")
	benthc		<- elt(aggregates, "benthc")
	benthclar	<- elt(aggregates, "benthclar")
	fishp		<- elt(aggregates, "fishp")
	fishplar	<- elt(aggregates, "fishplar")
	fishm		<- elt(aggregates, "fishm")
	fishd		<- elt(aggregates, "fishd")
	fishdlar	<- elt(aggregates, "fishdlar")
	bird		<- elt(aggregates, "bird")
	seal		<- elt(aggregates, "seal")
	ceta		<- elt(aggregates, "ceta")

	#Plot full time series of output

	par(mfrow=c(4,4))

	tsplot1("Total nitrogen",totalN)

	tsplot2("Susp. detritus",s_detritus,detritus_d)
	legend("topright",box.lty=0,bg="transparent",legend=c("Surface","Deep"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))


	tsplot1("Sediment detritus",x_detritus)

	tsplot2("Nitrate",s_nitrate,nitrate_d)
	legend("topright",box.lty=0,bg="transparent",c("Surface","Deep"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))


	tsplot1("Pore_w nitrate",x_nitrate)

	tsplot2("Ammonia",s_ammonia,ammonia_d)
	legend("topright",box.lty=0,bg="transparent",c("Surface","Deep"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	tsplot1("Pore_w ammonia",x_ammonia)

	tsplot2("Phytoplankton",s_phyt,phyt_d)
	legend("topright",box.lty=0,bg="transparent",c("Surface","Deep"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	tsplot1("Omniv. zoop",herb)
	tsplot1("Carniv. zoo",carn)

	#tsplot1("Susp/dep. benthos",benths)
	#tsplot1("Carniv. benthos",benthc)

	tsplot2("Susp/dep. benthos",benths,benthslar)
	legend("topright",box.lty=0,bg="transparent",c("Adults","Larvae"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	tsplot2("Carniv. benthos",benthc,benthclar)
	legend("topright",box.lty=0,bg="transparent",c("Adults","Larvae"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	#tsplot3("Pelagic fish",fishp,fishplar,fishm)
	tsplot2("Planktiv. fish",fishp,fishplar)
	#tsplot1("Pel. fish adults",fishp)
	legend("topright",box.lty=0,bg="transparent",c("Adults","Larvae"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	tsplot1("Migratory fish",fishm)

	tsplot2("Demersal fish",fishd,fishdlar)
	#tsplot1("Dem. fish adults",fishd)
	legend("topright",box.lty=0,bg="transparent",c("Adults","Larvae"),col=c("black","black"),lty=c(1,2),pt.cex=c(1,1),cex=c(0.9,0.9))

	#tsplot1("Birds/mammals (*10^-2)",bird*1e2)
	#tsplot2("Birds/mammals (*10^-2)",bird*1e2,rep(0,nrow(out)))
	tsplot3("Birds/mammals (*10^-2)",bird*1e2,seal*1e2,ceta*1e2)
	legend("topright",box.lty=0,bg="transparent",c("Birds","Pinnipeds","Cetaceans"),col=c("black","black","red"),lty=c(1,2,2),pt.cex=c(1,1,1),cex=c(0.8,0.8,0.8))

	#print("Black = birds, dashed = seals, red = cetaceans")

}


