#
# plot_time_series_annual_land_disc.R
#
#' Time series plots of annual integrated catches from the inshore and offshore zones
#'
#' Multi-panel time series bar-plots of annual integrated catch of each living guild in the inshore and offshore zones over the ful length of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series bar-plots of annual integrated catches
#' of the potentially catchable guilds in the ecology model:
#' macrophytes, carnivorous zooplankton, suspension/deposit and carnivore/scavenge feeding benthos, planktivorous, demersal and migratory fish,
#' birds, pinnipeds and cetaceans.
#'
#' Each panel represents a single guild. In each plot panel, the x-axis represents the sequence of years of the model run, and the y-axis is the annual integrated catch.
#' For each year there are two bars corresponding to the inshore and offshore zones. Each bar is subdivided into landed and discarded components ofthe catch.
#' Green and blue sections are offshore and inshore landings respectively; black and grey sections are offshore and inshore discards respectively.
#'
#' Units for the catch are mMN/m2/year
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_for_each_guild_catch_by_gear}} , \code{\link{plot_for_each_gear_catch_by_guild}} 
#'
#' @importFrom graphics barplot title
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the time series of annual landings and discards
#' plot_time_series_annual_land_disc(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_time_series_annual_land_disc(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_time_series_annual_land_disc(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the time series of annual landings and discards
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_time_series_annual_land_disc(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_time_series_annual_land_disc(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

plot_time_series_annual_land_disc <- function(model, results) {
    
start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	#----------------------------------------------------------------
	#In each plot panel:
	#    black and grey are offshore and inshore discards respectively
	#    green and blue are offshore and inshore landings respectively
	#----------------------------------------------------------------

	total.annual.catch <- elt(results, "total.annual.catch")

        ngroups<-10 # this is the number of groups in gear_group_proportions

        offshore_annual_group_land_disc <- elt(total.annual.catch, "offshore_annual_group_land_disc")
        inshore_annual_group_land_disc  <- elt(total.annual.catch, "inshore_annual_group_land_disc")

	group_names <- c(
		"Planktivorous fish",
		"Quota-limited Demersal fish",
		"Non-quota demersal fish",
		"Migratory fish",
		"Susp/dep feeding benthos",
		"Carn/scav feeding benthos",
		"Pelagic invertebrates",
		"Birds",
		"Pinnipeds",
		"Cetaceans","Macrophytes"
	)

	par(mfrow=c(6,2))
	par(mar=c(4,5,1,0.5))


	for(dsa in 1:(ngroups+1)) {

		mt<-group_names[dsa]

		offshore_data2plot<-rbind(offshore_annual_group_land_disc[,(ngroups+2)+dsa],offshore_annual_group_land_disc[,1+dsa])
		inshore_data2plot<-rbind(inshore_annual_group_land_disc[,(ngroups+2)+dsa],inshore_annual_group_land_disc[,1+dsa])

		offshore_catch<-colSums(offshore_data2plot)
		inshore_catch<-colSums(inshore_data2plot)

		colnames(offshore_data2plot)<-offshore_annual_group_land_disc[,1]

		barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,0.0000001+(1.2*max(c(offshore_catch,inshore_catch)))),xlim=c(0,(ncol(offshore_data2plot)+1)),width=rep(0.4,ncol(offshore_data2plot)), space=c(0.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",ann=FALSE,cex.axis=0.9)
		axis(side=2,las=1,cex.axis=0.9)
		if(dsa==1)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==3)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==5)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==7)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==0)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==11)  mtext("Catch",cex=0.7,side=2,line=3.5)
		#if(dsa==7)  mtext("Years",cex=0.7,side=1,line=2)
		#if(dsa==8)  mtext("Years",cex=0.7,side=1,line=2)
		#if(dsa==9)  mtext("Years",cex=0.7,side=1,line=2)
		#if(dsa==10)  mtext("Years",cex=0.7,side=1,line=2)
		if(dsa==11)  mtext("Years",cex=0.7,side=1,line=2)
		if(dsa==12)  mtext("Years",cex=0.7,side=1,line=2)
		title(main=mt,cex.main=1.0)
		barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,ncol(offshore_data2plot)), space=c(1.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",xaxt="n",ann=FALSE)
	}


	#Calculate total landings and disards avross all guilds
	offshore_data2plot<-rbind(rowSums(offshore_annual_group_land_disc[,((ngroups+2)+1) : ((ngroups+2)+(ngroups+1))]),rowSums(offshore_annual_group_land_disc[,2:(ngroups+2)]))
	inshore_data2plot<- rbind(rowSums( inshore_annual_group_land_disc[,((ngroups+2)+1) : ((ngroups+2)+(ngroups+1))]),rowSums( inshore_annual_group_land_disc[,2:(ngroups+2)]))

	offshore_catch<-colSums(offshore_data2plot)
	inshore_catch<-colSums(inshore_data2plot)

	colnames(offshore_data2plot)<-offshore_annual_group_land_disc[,1]

	mt<-"All guilds combined"

	barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,0.0000001+(1.2*max(c(offshore_catch,inshore_catch)))),xlim=c(0,(ncol(offshore_data2plot)+1)),width=rep(0.4,ncol(offshore_data2plot)), space=c(0.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",ann=FALSE,cex.axis=0.9)
	axis(side=2,las=1,cex.axis=0.9)
	mtext("Years",cex=0.7,side=1,line=2)
	title(main=mt,cex.main=1.0)
	barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,ncol(offshore_data2plot)), space=c(1.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",xaxt="n",ann=FALSE)
}


