#
# Monte_Carlo_StrathE2E.R
#
#' Runs a Monte Carlo simulation with StrathE2E and derives centiles of credible values of model outputs
#'
#' The Monte Carlo scheme generates an ensemble of model runs by sampling the ecology model parameters from uniform distibutions
#' centred on a maximum likelihood parameter set. The latter needs to be arrived at by prior application the various simulated
#' annealing functions in the package to 'fit' the model to a suite of observational data on the state of ecosystem
#' (target data for fitting are in /Modelname/Variantname/Target_data/annual_target_data_*.csv).
#' Each iteration in the Monte Carlo scheme then generates a unique time series of model outputs at daily intervals,
#' together with the overall likelihood of the obervational target data. 
#'
#' The function should be initialised with starting values, parameter and driving data configuration corresponding to a stationary
#' state with the maximum likelihood of the observational target data. This configuration needs to be loaded by a prior call of the
#' read_model() function. This represents the baseline for sampling of the ecology model parameters.
#'
#' The coefficients of variation for jiggling the ecology parameter can be varied in real-time
#' during the run by editing the file "CredIntSim_SD.csv" in the folder
#' /Parameters/Parameter_SD_control/ of the model version. However, it is recommended to leave the setting constant for the duration of a run.
#' A cv of 0.10 to 0.15 is recommended. If comparing the credible intervals for two versions of a model then it is important to use the same cv in both cases.
#'
#' On completion of all the iterations of the model, for each model output variable in turn, values from the individual runs and
#' their associated model likelihoods are assembled as a list of paired values. The list is then sorted by ascending values of the model variable,
#' and the cumulative likelihoods with increasing value of model variable is calculated. Values for the model variable at standard proportions
#' (0.005, 0.25, 0.5, 0.75, 0.995) of the maximum cumulative likelhood are then extracted by interpolation.
#' These represent the credible intervals of model output given uncertainty in the ecology model parameters.
#'
#' Outputs from the Monte_Carlo_StrathE2E() function are stored in csv files in the folder /results/Modelname/Variantname/CredInt/ with an
#' identifier for the simulation (model.ident) created by the read_model() function. There are two types of output. First is simple an
#' accumulation of all the standard outputs from StrathE2E on a run-by-run basis. The output of these data is updated every 10 iterations
#' for security to mitigate against data loss in the event of an unforseen interruption of a simulation. The second type of output is the results
#' of post-processing the raw data to generate the centile of the distribution of credible values.
#'
#' The function displays various real-time graphics to show the progress of the simulation. During the StrathE2E-running phase of the
#' process an x-y graph is updated after each iteration (x-axis=iterations, y-axis=Likelihood of the target data), with a horizontal grey line showing the
#' baseline (maximum liklihood) result, and black symbols showing the likelihood for each iteration based on the parameter values sampled from the baseline.
#' The y-axis limits for the graph can be varied in real-time
#' during the run by editing the file "CredIntSim_SD.csv" in the folder /Parameters/Parameter_SD_control/ of the model version.
#'
#' During the post-processing phase, an x-y graph of the cumulative
#' likelihood (y-axis) vs output variable values, is displayed as the code works through all of the output variables.
#'
#' WARNING - the scheme can take a long time to run (~2 days with the default settings), and generate large output files.
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param nyears Number of years to run each instance of the model. Needs to be long enough to allow the model to attain a stationary state (default=50)
#' @param n_iter Number of iterations of the model (default=1000)
#'
#' @return Real-time graphical displays during the simulation, and CSV files of output and post-processed data
#'
#' @seealso \code{\link{read_model}}, \code{\link{plot_final_year_time_series_data_with_credible_intervals}}, \code{\link{plot_final_year_migration_data_with_credible_intervals}} , \code{\link{plot_inshore_vs_offshore_anavmass_with_credible_intervals}} 
#'
#' @importFrom stats runif
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "1970-1999")
#'
#' # Run the Monte Carlo process and generate the results files
#' # WARNING: This run will take more than 24h to complete !
#' Monte_Carlo_StrathE2E(model,nyears=50,n_iter=1000)
#'
#' # A quick demonstration run:
#' model <- read_model("North_Sea", "1970-1999", model.ident="Demo")
#' Monte_Carlo_StrathE2E(model,nyears=5,n_iter=10)
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

Monte_Carlo_StrathE2E <- function(model, nyears=50, n_iter=1000) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	setup			<- elt(model, "setup")
	model.path		<- elt(setup, "model.path")
	resultsdir		<- elt(setup, "resultsdir")
	identifier		<- elt(setup, "model.ident")

	data			<- elt(model, "data")
	fitted.parameters	<- elt(data, "fitted.parameters")

	credpath		<- makepath(resultsdir, CREDINT_DIR)

	print(date())

#	n_iter			<- 1000		# Maximum number of iterations of the parameter randomisation process
#	n_iter			<- 10		# ZZ temp

	datastore <- c(
		fitted.parameters,
		annual_obj = 1e-60
	)

	for (kkk in 1:n_iter){

		cat("Iteration =", kkk, "\n")

		annealing.parms <- read_SD_credint(model.path)		# read every loop so control is available during a run
		if (kkk==1) annealing.parms[] <- 0.0

		perturbed <- perturb_parameters_all(datastore, annealing.parms)

		# perturbed parameters built into model.parameters for run except for the first run  :
		if(kkk>1) model$data$fitted.parameters <- perturbed[1:(length(perturbed)-1)]
		results <- StrathE2E(model, nyears=nyears)


		opt_results <- elt(results, "final.year.outputs", "opt_results")
		annual_obj <- elt(results, "final.year.outputs", "annual_obj")

		#Merge derived outputs with iteration number and liklihood

		if(kkk==1) Results_to_store<-c(iteration=kkk,datastore[1:(length(datastore)-1)],likelihood=annual_obj)
		if(kkk>1) Results_to_store<-c(iteration=kkk,perturbed[1:(length(perturbed)-1)],likelihood=annual_obj)

		opt_results_to_store<-c(iteration=kkk,annual_obj,opt_results[,3])
		names(opt_results_to_store)<-c("iteration","likelihood",as.vector(opt_results$Description))

		monthlyfinal <- elt(results, "final.year.outputs", "monthly.averages")
		monthlyfinal$month<-seq(1,12)
		monthlyfinal$iteration<-kkk
		monthlyfinal$likelihood<-annual_obj

		#-------------------------------------------------------------------------------------------------------

		#Extract the final year of model output and volumetric data to be saved for later analysis
		lastyear <- extract_final_year_of_run_plus_volumetric_data(model, results)

		lastyear$iteration<-kkk
		lastyear$likelihood<-annual_obj

		#--------------------------------------------------------------------------------

		mass_results <- elt(results, "final.year.outputs", "mass_results_inshore")
		annual_flux_results <- elt(results, "final.year.outputs", "annual_flux_results_inshore")
		inshoreaamass<-mass_results[,1]
		inshoreaamass<-c(kkk,annual_obj,inshoreaamass)
		names(inshoreaamass)<-c("iteration","likelihood",mass_results$Description)
		inshore_annual_flux<-annual_flux_results[,1]
		inshore_annual_flux<-c(kkk,annual_obj,inshore_annual_flux)
		names(inshore_annual_flux)<-c("iteration","likelihood",annual_flux_results$Description)

		mass_results <- elt(results, "final.year.outputs", "mass_results_offshore")
		annual_flux_results <- elt(results, "final.year.outputs", "annual_flux_results_offshore")
		offshoreaamass<-mass_results[,1]
		offshoreaamass<-c(kkk,annual_obj,offshoreaamass)
		names(offshoreaamass)<-c("iteration","likelihood",mass_results$Description)
		offshore_annual_flux<-annual_flux_results[,1]
		offshore_annual_flux<-c(kkk,annual_obj,offshore_annual_flux)
		names(offshore_annual_flux)<-c("iteration","likelihood",annual_flux_results$Description)

		mass_results <- elt(results, "final.year.outputs", "mass_results_wholedomain")
		annual_flux_results <- elt(results, "final.year.outputs", "annual_flux_results_wholedomain")
		wholeaamass<-mass_results[,1]
		wholeaamass<-c(kkk,annual_obj,wholeaamass)
		names(wholeaamass)<-c("iteration","likelihood",mass_results$Description)
		whole_annual_flux<-annual_flux_results[,1]
		whole_annual_flux<-c(kkk,annual_obj,whole_annual_flux)
		names(whole_annual_flux)<-c("iteration","likelihood",annual_flux_results$Description)

		networkdata_results <- elt(results, "final.year.outputs", "NetworkIndexResults")
		networkdata<-networkdata_results[,1]
		networkdata<-c(kkk,annual_obj,networkdata)
		names(networkdata)<-c("iteration","likelihood",rownames(networkdata_results))
 

		#Now add the new sets of results to the previous ones if kkk>1 or set up the results store if kkk=1

		if(kkk==1) {

			lastyearstore<-lastyear

			parameterstore<-data.frame(Results_to_store[1])
			for(jjjj in 2:length(Results_to_store)) {
				parameterstore[,jjjj]<-Results_to_store[jjjj]
			}
			names(parameterstore)<-names(Results_to_store)
			rownames(parameterstore)<-NULL

			optresultsstore<-data.frame(opt_results_to_store[1])
			for(jjjj in 2:length(opt_results_to_store)) {
				optresultsstore[,jjjj]<-opt_results_to_store[jjjj]
			}
			names(optresultsstore)<-names(opt_results_to_store)
			rownames(optresultsstore)<-NULL

			monthlystore<-monthlyfinal

			inshoreaamassstore<-data.frame(inshoreaamass[1])
			for(jjjj in 2:length(inshoreaamass)) {
				inshoreaamassstore[,jjjj]<-inshoreaamass[jjjj]
			}
			names(inshoreaamassstore)<-names(inshoreaamass)
			rownames(inshoreaamassstore)<-NULL

			offshoreaamassstore<-data.frame(offshoreaamass[1])
			for(jjjj in 2:length(offshoreaamass)) {
				offshoreaamassstore[,jjjj]<-offshoreaamass[jjjj]
			}
			names(offshoreaamassstore)<-names(offshoreaamass)
			rownames(offshoreaamassstore)<-NULL

			wholeaamassstore<-data.frame(wholeaamass[1])
			for(jjjj in 2:length(wholeaamass)) {
				wholeaamassstore[,jjjj]<-wholeaamass[jjjj]
			}
			names(wholeaamassstore)<-names(wholeaamass)
			rownames(wholeaamassstore)<-NULL

			inshoreannualfluxstore<-data.frame(inshore_annual_flux[1])
			for(jjjj in 2:length(inshore_annual_flux)) {
				inshoreannualfluxstore[,jjjj]<-inshore_annual_flux[jjjj]
			}
			names(inshoreannualfluxstore)<-names(inshore_annual_flux)
			rownames(inshoreannualfluxstore)<-NULL

			offshoreannualfluxstore<-data.frame(offshore_annual_flux[1])
			for(jjjj in 2:length(offshore_annual_flux)) {
				offshoreannualfluxstore[,jjjj]<-offshore_annual_flux[jjjj]
			}
			names(offshoreannualfluxstore)<-names(offshore_annual_flux)
			rownames(offshoreannualfluxstore)<-NULL

			wholeannualfluxstore<-data.frame(whole_annual_flux[1])
			for(jjjj in 2:length(whole_annual_flux)) {
				wholeannualfluxstore[,jjjj]<-whole_annual_flux[jjjj]
			}
			names(wholeannualfluxstore)<-names(whole_annual_flux)
			rownames(wholeannualfluxstore)<-NULL

			networkstore<-data.frame(networkdata[1])
			for(jjjj in 2:length(networkdata)) {
				networkstore[,jjjj]<-networkdata[jjjj]
			}
			names(networkstore)<-names(networkdata)
			rownames(networkstore)<-NULL
		}

		if(kkk>1) {
			lastyearstore<-rbind(lastyearstore,lastyear)
			parameterstore<-rbind(parameterstore,Results_to_store)
			optresultsstore<-rbind(optresultsstore,opt_results_to_store)
			monthlystore<-rbind(monthlystore,monthlyfinal)
			inshoreaamassstore<-rbind(inshoreaamassstore,inshoreaamass)
			offshoreaamassstore<-rbind(offshoreaamassstore,offshoreaamass)
			wholeaamassstore<-rbind(wholeaamassstore,wholeaamass)
			inshoreannualfluxstore<-rbind(inshoreannualfluxstore,inshore_annual_flux)
			offshoreannualfluxstore<-rbind(offshoreannualfluxstore,offshore_annual_flux)
			wholeannualfluxstore<-rbind(wholeannualfluxstore,whole_annual_flux)
			networkstore<-rbind(networkstore,networkdata)
		}



		#-------------------------------------------------------------------

		#Plot or update the time series of proposal and accepted likelihoods so far....
		if(kkk>1){
			par(mfrow=c(1,1))
			axmin <- elt(annealing.parms, "axmin")
			axmax <- elt(annealing.parms, "axmax")
			plot(c(1,nrow(optresultsstore)),rep(optresultsstore$likelihood[1],2),ylim=c(axmin,axmax),xlim=c(1,kkk),xlab="Iterations",ylab="Target data likelihood",type="l",lwd=3,col="grey")
			points(optresultsstore$iteration,optresultsstore$likelihood,type="l",col="black")
#                	legend("topleft",c("baseline","samples"),bg="transparent",
#				col=c("grey","black"),
#				lty=c(1,1),lwd=c(3,1),
#				pt.cex=c(1,1))
                	legend("topleft",c("baseline","samples"),bg="transparent",
				col=c("grey","black"),
				lty=c(1,0),lwd=c(3,0),
				pch=c(NA,20),
				pt.cex=c(0,1))

		}

		#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		#Output the accumulated output to disc every 10 iterations for security

		if(floor(kkk/10)==kkk/10) {
			writecsv(lastyearstore,			csvname(credpath, "CredInt_cumulative_lastyear", identifier),		row.names=FALSE)
			writecsv(parameterstore,		csvname(credpath, "CredInt_cumulative_parameters",identifier),		row.names=FALSE)
			writecsv(optresultsstore,		csvname(credpath, "CredInt_cumulative_targetresults",identifier),		row.names=FALSE)
			writecsv(monthlystore,			csvname(credpath, "CredInt_cumulative_monthly",identifier),		row.names=FALSE)
			writecsv(inshoreaamassstore,		csvname(credpath, "CredInt_cumulative_inshoreaamass",identifier),		row.names=FALSE)
			writecsv(offshoreaamassstore,		csvname(credpath, "CredInt_cumulative_offshoreaamass",identifier),		row.names=FALSE)
			writecsv(wholeaamassstore,		csvname(credpath, "CredInt_cumulative_wholeaamass",identifier),		row.names=FALSE)
			writecsv(inshoreannualfluxstore,	csvname(credpath, "CredInt_cumulative_inshoreannualflux",identifier),	row.names=FALSE)
			writecsv(offshoreannualfluxstore,	csvname(credpath, "CredInt_cumulative_offshoreannualflux",identifier),	row.names=FALSE)
			writecsv(wholeannualfluxstore,		csvname(credpath, "CredInt_cumulative_wholeannualflux",identifier),	row.names=FALSE)
			writecsv(networkstore,			csvname(credpath, "CredInt_cumulative_network",identifier),		row.names=FALSE)
		}

	}

	#Output the final accumulated output to disc 

	writecsv(lastyearstore,			csvname(credpath, "CredInt_cumulative_lastyear", identifier),		row.names=FALSE)
	writecsv(parameterstore,		csvname(credpath, "CredInt_cumulative_parameter",identifier),		row.names=FALSE)
	writecsv(optresultsstore,		csvname(credpath, "CredInt_cumulative_targetresults",identifier),		row.names=FALSE)
	writecsv(monthlystore,			csvname(credpath, "CredInt_cumulative_monthly",identifier),		row.names=FALSE)
	writecsv(inshoreaamassstore,		csvname(credpath, "CredInt_cumulative_inshoreaamass",identifier),		row.names=FALSE)
	writecsv(offshoreaamassstore,		csvname(credpath, "CredInt_cumulative_offshoreaamass",identifier),		row.names=FALSE)
	writecsv(wholeaamassstore,		csvname(credpath, "CredInt_cumulative_wholeaamass",identifier),		row.names=FALSE)
	writecsv(inshoreannualfluxstore,	csvname(credpath, "CredInt_cumulative_inshoreannualflux",identifier),	row.names=FALSE)
	writecsv(offshoreannualfluxstore,	csvname(credpath, "CredInt_cumulative_offshoreannualflux",identifier),	row.names=FALSE)
	writecsv(wholeannualfluxstore,		csvname(credpath, "CredInt_cumulative_wholeannualflux",identifier),	row.names=FALSE)
	writecsv(networkstore,			csvname(credpath, "CredInt_cumulative_network",identifier),		row.names=FALSE)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


	#Now process the stored outputs to create the credible intervals for all of the model outputs

	CredInt_make_daily_results(model, lastyearstore)
	CredInt_make_daily_migration_results(model, lastyearstore)
	CredInt_make_daily_flux_results(model, lastyearstore)
	CredInt_make_aamass_results(model, inshoreaamassstore, offshoreaamassstore, wholeaamassstore)
	CredInt_make_annual_flux_results(model, inshoreannualfluxstore, offshoreannualfluxstore, wholeannualfluxstore)
	CredInt_make_monthly_results(model, monthlystore)
	CredInt_make_network_results(model, networkstore)
	CredInt_make_target_results(model, optresultsstore)
	CredInt_make_parameter_results(model, parameterstore)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

}
