#
# annealing_find_gear_activity_mult_HR_target.R
#
#' Optimise fishing gear activity multipliers to maximum the likelihood of known harvest ratios
#'
#' Launches a simulated annealing process to find the set of fishing fleet model gear activity multipliers
#' producing the maximum likelihood of a set of known harvest ratios using only the fishing fleet model,
#' given specified effort-harvest ratio scaling parameters.
#' Note that there is a related function annealing_find_gear_activity_mult_ecosystem_target() which uses the ecology model to
#' find the activity multipliers which maximise the likelihood of observed target data on the state of the ecosytem, given values of the
#' ecology model parameters and the scaling values linking effort to harvest ratio.
#'
#' The known harvest ratios which act as the target for the optimization process are located in the file
#' ../Modelname/Variantname/Target_data/fishing_fleet_target_harvest_ratios*.csv, where * is an identifier.
#'
#' Simulated annealing is an iterative random-walk type process which searches the parameter space
#' of a model to locate the combination which maximises the likelihood of a set of observed
#' data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
#' may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
#' avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
#' by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1). The task here is a bit harder than normal
#' because of the potentially large overlap in the selectivity patterns of the fishing gears with respect to the living guilds in the ecosystem. So there are
#' some non-standard features to try and avoid local maxima.
#'
#' Model configuration and initial values of the ecology model parameters need to be
#' assembled by a prior call of the read_model() function.
#' 
#' NOTE that the user.path argument in the read_model() function call needs to point to a user workspace folder, not the default
#' North Sea model provided with the package. This is because the annealing function needs write-access to the model /Parameters folder,
#' but the /extdata/Models folder in the package installation is read-only.
#' To use the annealing function on the North Sea model, use the copy_model() function to make a copy of the
#' North Sea model in the user workspace.
#'
#' The initial coefficient of variation for searching the parameter space is set by a function argument. The CV decreases in steps with increasing iterations.
#' The function repeats the iteration process multiple times. The repeats are referred to as 'chains'. Each chain follows a different pathway through the parameter space.
#' At the end of the process, the best-fit set of activity multipliers is selected from across all the chains.
#'
#' To cope with the overlap in selectivity of the fishing gears, the function uses a set of linkage parameters specified in the the file /Parameters/fishing_fleet_gear_linkages.csv
#' These parameters force selected gears activities to vary in concert +/- some variation, rather than independently. The parameters for the gear linkages are located in the file
#' ../Modelname/Variantname/Parameters/fishing_fleet_gear_linkages.csv. The table of linkages specifies which gear activity rates are forced to 
#' vary in concert during the fitting process, as opposed to varying independently. The value of the linkage coefficient defines the 
#' scaling of changes in the activity rate of a dependent gear relative to its linked independent gear. For example, if gear 8 is permitted to vary 
#' independently (value in column "Gear to which linked" = NA and "Linkage coefficient" = NA). If gear 9 is dependent on gear 8 then the activity rate 
#' of gear 9 this would be specified by e.g. "Gear to which linked" = 8 and "Linkage coefficient" = 0.645. This would force the activity of gear 9 to be 
#' set at gear8_activity * (0.645 +/- a random variation defined by the argument deltaGrat). 
#'
#' The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
#' the likelihoods of the proposed and accepted parameter sets at each iteration.
#' Y-axis (likelihood of the target data) range of the real time plot can be varied during the run by 
#' editing the setup file "annealing_SD_fishing.csv"
#'
#' Once all the chains have been completed the function produces some diagnostic box and whisker plots showing the uncertainty in the estimated activity rate multipliers
#' and in the estimated harvest ratios across the end-points of all the chains. In each diagnostic plot the maximum likelihood model resullts are shown by a red
#' bar, and the distribution of all results by black box and whiskers.
#'
#' At the end of the procedure a new version of the gear activity multipliers file is exported
#' to the folder /Parameters of the model version, with a user defined identifier specified by the model.ident argument
#' in the read_model() function. The histories of proposed and accepted parameter combinations
#' are saved as CSV files in the results folder.
#'
#' To preserve the new activity multipliers and incorporate them into the fishing fleet model parameterisation
#' the multiplier values need to be applied to the activity rates specified in the data input file /Parameters/fishing_activity_parameters_*.csv.
#' Manually update the values in fishing_activity_parameters_*.csv, by multiplying the existing values by the new multipliers emerging from the annealing process.
#'
#' If the edited file fishing_activity_parameters_*.csv is saved with a new identifier (*) then in order to use it in a subsequent run of the
#' StrathE2E model (using the StrathE2E() function) it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new file.
#'
#' @param model R-list object generated by the read_model() function which defined the model configuration
#' @param nchains Number of repeats (chains) of the simulated annealing process (default=100)
#' @param n_iter Number of iterations of the fishing fleet model per chain (default=3000)
#' @param start_temperature Initial value of the simulated annealing temperature parameter (default=0.5). Suggested values in the range 0.0005 - 5. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood
#' @param cooling Rate at which the simulated annealing temperature declines with iterations (default=0.985). Suggested values in the range 0.9 - 0.985
#' @param deltaHi Initial coefficient of variation for jiggling the activity multiplier values (default=0.2)
#' @param attenuationstep Number of itereations between down-steps of the jiggling factor applied to the multiplier values. The jiggling rate is attenuated by a factor of 2 every xx iterations (default=500)
#' @param deltaGrat Coefficient of variation for jiggling the gear linkage values (default=0.25)
#'
#' @return CSV file of fitted gear activity multipliers and statistics of the likelihood weighted distributions of activity multipliers and the fit to harvest ratios
#'
#' @seealso \code{\link{list_models}} , \code{\link{read_model}} , \code{\link{StrathE2E}} , \code{\link{annealing_find_ecology_parameters}} , \code{\link{annealing_find_harvest_ratio_mult}} , \code{\link{annealing_find_gear_activity_mult_ecosystem_target}}
#'
#' @importFrom stats runif
#'
#' @export
#'
#' @examples
#' #Copy the 1970-1999 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#' copy_model("North_Sea", "1970-1999",
#'            dest.path="C:/Users/username/Documents/Models")
#'
#' # Load the 1970-1999 version of the North Sea model from the user workspace:
#' model<-read_model(model.name="North_Sea",
#'                   model.variant="1970-1999",
#'                   model.ident="TEST",
#'                   user.path="C:/Users/username/Documents/Models")
#'
#' # Quick Demo of the annealing function in operation:
#' annealing_find_gear_activity_mult_HR_target(model, nchains=5, n_iter=100, start_temperature=0.5)
#' # More realistic configuration would be (WARNING - this will an hour or so to run) :
#' annealing_find_gear_activity_mult_HR_target(model, nchains=100, n_iter=3000, start_temperature=0.5)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------


annealing_find_gear_activity_mult_HR_target <- function(model, nchains=100, n_iter=3000, start_temperature=0.5, cooling=0.985, deltaHi=0.20, attenuationstep=500, deltaGrat=0.25) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	setup				<- elt(model, "setup")
	read.only			<- elt(setup, "read.only")
	model.path			<- elt(model, "setup", "model.path")
	model.ident			<- elt(model, "setup", "model.ident")
	resultsdir			<- elt(model, "setup", "resultsdir")

	data				<- elt(model, "data")
	fleet.model			<- elt(data, "fleet.model")
	gear_codes			<- elt(fleet.model, "gear_codes")
	gear_mult			<- elt(fleet.model, "gear_mult")
	gear_activity			<- elt(fleet.model, "gear_activity")
	gear_group_rel_power		<- elt(fleet.model, "gear_group_rel_power")
	HRscale_vector_multiplier	<- elt(fleet.model, "HRscale_vector_multiplier")

	if (read.only) {
		cat("Warning: cannot write fitted parameters back to the model input folders - model is read-only\n")
		cat("Warning: to fix this, make a copy of the model using copy_model() into your own workspace.\n")
		stop("Model is not writable!")
	}

	print(date())


	if(length(which(HRscale_vector_multiplier<1|HRscale_vector_multiplier>1))>0) {
		print("**************************************************************************")
		print("WARNING - one or more baseline Harvest Ratio scaling values differs from 1")
		print("**************************************************************************")
	}

	#End of model setup

	#Read the gear linkages and target data
	gear_linkages	<- get.model.file(model.path, PARAMETERS_DIR, file.pattern=FISHING_FLEET_GEAR_LINKAGES)
	target_HRs	<- get.model.file(model.path, TARGET_DATA_DIR, file.pattern=FISHING_FLEET_TARGET_HR)

	fleet_vector_target<-c(
		target_HRs[1,2],target_HRs[1,3],
		target_HRs[2,2],target_HRs[2,3],
		target_HRs[3,2],target_HRs[3,3],
		target_HRs[4,2],target_HRs[4,3],
		target_HRs[5,2],target_HRs[5,3],
		target_HRs[6,2],target_HRs[6,3],
		target_HRs[7,2],target_HRs[7,3],
		target_HRs[8,2],target_HRs[8,3],
		target_HRs[9,2],target_HRs[9,3],
		target_HRs[10,2],target_HRs[10,3]
	)

	ignoredata<-which(target_HRs[,4]==0)
	if(length(ignoredata)>0){
		for(jjjj in 1:length(ignoredata)){
			if(jjjj==1) ignorevector<-c( (((ignoredata[jjjj]-1)*2)+1),(((ignoredata[jjjj]-1)*2)+2) )
			if(jjjj>1) ignorevector<-c(ignorevector, (((ignoredata[jjjj]-1)*2)+1),(((ignoredata[jjjj]-1)*2)+2) )
		}
		fleet_vector_target[ignorevector]<-NA
	}

	MAINstore<-data.frame(G1=0,G2=0,G3=0,G4=0,G5=0,G6=0,G7=0,G8=0,G9=0,G10=0,G11=0,G12=0,lik=0)
	HRstore<-data.frame(PF1=0,PFo=0,DF1=0,DFo=0,MFi=0,MFo=0,BSi=0,BSo=0,Bci=0,BCo=0,Czi=0,Czo=0,BD_i=0,BD_o=0,SL_i=0,SL_o=0,CT_i=0,CT_o=0,KP_i=0,KP_o=0,lik=0)

	for(chain in 1:nchains){

		temperature<-start_temperature 

		#if(chain>1) source("library/PREPARE_THE_FISHING_FLEET_MODEL_tp_KELP.R")

		#source(paste("library/fishing/configure_fishing_fleet_model_MERP_NEW_6HABS_ROCK_SQUID2_fromfile_BMbycatch_selSW_NOREAD_CLEAN_OFFAL_tp_KELP.R",sep=""))

		gear_mult_START<-gear_mult

		#Set up a dataframe to hold the proposed and accepted gear_multipliers

		GEARmult_proposal<-data.frame(G1=0,G2=0,G3=0,G4=0,G5=0,G6=0,G7=0,G8=0,G9=0,G10=0,G11=0,G12=0,lik=0)
		GEARmult_proposal[1,1:12]<-gear_mult_START
		GEARmult_accepted<-GEARmult_proposal

		acceptedstore_annual_obj<-1e-60
		proposalstore_annual_obj<-1e-60

		n_acceptances<-0   # Counter for the number of parameter acceptances which have occurred

		#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		#ITERATE THE ANNEALING PROCESS.....

		lastbest<-1e-60
		for (kkk in 1:n_iter){

			if(kkk>0) deltaH<-deltaHi
			if(kkk>(attenuationstep)) deltaH<-deltaHi/2
			if(kkk>(attenuationstep*2)) deltaH<-deltaHi/4
			if(kkk>(attenuationstep*3)) deltaH<-deltaHi/8
			if(kkk>(attenuationstep*4)) deltaH<-deltaHi/16

			#FIRST RUN THROUGH THE MODEL WITH THE INITIAL GEAR ACTIVITY RATES.....

			if(kkk>1){

				#Jiggle all of the last accepted gear mults...
				for(www in 1:12){ 
     					gear_mult[www] <- max(0,rnorm(1,GEARmult_accepted[(kkk-1),www],deltaH*GEARmult_accepted[(kkk-1),www]))
				}

				#Now skim through and over-ride the assigned mult values for gears that are linked to another...
				for(www in 1:12){
					if(is.na(gear_linkages[www,3])==FALSE){
						link_gear<-gear_linkages[www,3]
						link_coef<-gear_linkages[www,4]
						gear_mult[www]<-gear_mult[link_gear] * max(0,rnorm(1,link_coef,deltaGrat*link_coef))
					}
				}

				GEARmult_proposal[kkk,]<-c(gear_mult,0)
				GEARmult_accepted[kkk,]<-GEARmult_accepted[(kkk-1),]
			}    

			model$data$fleet.model$gear_mult <- gear_mult

			build <- build_model(model)

			#Run the fleet model to create the vector 'fleet_vector' 
			#fleet_model_output<-FLEET_MODEL(gear_activity,
                                #gear_group_rel_power,
                                #gear_group_discard,
                                #gear_group_gutting,
                                #gear_ploughing_rate,
                                #gear_habitat_activity,
                                #HRscale_vector,
                                #gear_mult,
                                #quota_nonquota_parms_vector,
                                #DFsize_SWITCH,
                                #DFdiscard_SWITCH,
                                #plough_depth_vector,
                                #BSmort_gear, BCmort_gear,
                                #offal_prop_live_weight,
                                #habitat_areas,
                                #0,
                                #0,0,0) 

			#Grab the vector of parameter values that needs to be included in the ecology model input
			#fleet_vector<-((as.data.frame(fleet_model_output[1]))[,1])

			#fleet_vector_proposal<-fleet_vector[1:20]
			fleet_vector <- elt(build, "fleet.output", "fleet_vector")
			fleet_vector_proposal <- fleet_vector[1:20]

			#Now calculate a likelihood for the target HRs

			HRtol<-0.2*fleet_vector_target

			chi <- ((fleet_vector_target-fleet_vector_proposal)^2) / (2*(HRtol^2))

			validtarg<-length(which(is.na(fleet_vector_target)==FALSE))
			p <- exp(-1*( (sum(chi[1:20],na.rm=TRUE))/validtarg) )  # excludes THE CARNZOO HARVEST RATIO

			proposalstore_annual_obj<-c(proposalstore_annual_obj,p)
			acceptedstore_annual_obj<-c(acceptedstore_annual_obj,acceptedstore_annual_obj[length(acceptedstore_annual_obj)])
		        GEARmult_proposal[kkk,13]<-p

			#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

			temperature<-temperature*cooling

			#-------------------------------------------------------------------
			#Now the Metropolis algorithm.....

			lik_ratio<-exp(((log(p)) - log(lastbest))/temperature)

			rand<-runif(1,0,1)

			#--------------------------------

			if(lik_ratio>rand){
				n_acceptances<-n_acceptances+1
				GEARmult_accepted[kkk,]<-GEARmult_proposal[(kkk),]
				fleet_vector_accepted<-fleet_vector_proposal
				lastbest<-p
			}

			#--------------------------------

			#Plot or update the time series of proposal and acepted likelihoods so far....
			if(kkk>1){

				axmin<-0
				axmax<-1

				par(mfrow=c(1,1))

				#Plot the basemodel results 
				plot(seq(1,nrow(GEARmult_proposal)),GEARmult_proposal[,13],ylim=c(axmin,axmax),xlim=c(1,kkk),xlab="Iterations",ylab="Target data likelihood",type="l",col="grey",main=(paste("Chain ",chain,sep="")))
				points(seq(1,nrow(GEARmult_accepted)),GEARmult_accepted[,13],type="l",col="black",lwd=3)
                		legend("topleft",c("accepted","proposed"),bg="transparent",col=c("black","grey"),lty=c(1,1),lwd=c(3,1),pt.cex=c(1,1))
			}


		}


		#-------------------------------------------------------------------


		MAINstore[chain,]<-GEARmult_accepted[nrow(GEARmult_accepted),]
		HRstore[chain,]<-c(fleet_vector_accepted, GEARmult_accepted[(nrow(GEARmult_accepted)),13] )


	}


	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


	#Now Process the results from all of the chains to extract the best fit and the credible intervals


	#-------------------------------------------------------
	
	#Find the best overall fit values
	gearmult_BEST<-MAINstore[which(MAINstore[,13]==max(MAINstore[,13])),]
	HRvalues_BEST<-HRstore[which(HRstore[,21]==max(HRstore[,21])),]


	#-------------------------------------------------------

	#set up a dataframe to hold the credible intervals of the gearmult values
	gearmult_credints<-array(dim=c(5,12),0)
	HRvalues_credints<-array(dim=c(5,20),0)
	guild_list<-c("PFi","PFo","DFi","DFo","MFi","MFo","Bsdi","Bsdo","Bcsi","Bcso","CZi","CZo",
              	"BDi","BDo","SLi","SLo","CTi","CTo","KPi","KPo")

   	colnames(gearmult_credints)<-gear_codes	# ZZ was gear_code_list # gear codes picked up from the initial gear activity input file
   	colnames(HRvalues_credints)<-guild_list

	#-------------------------------------------------------

	#Find the credible intervals of the distributions of gearmults and estimated HR values

	creds<-c(0.005,0.25,0.5,0.75,0.995)  

	for(gearid in 1:12){

		Result<-MAINstore[,gearid]
		Likelihood<-MAINstore[,13]

		gearmult_credints[,gearid]<-GetCredInt(Result,Likelihood,creds,var="",plotgraph=FALSE)
		#dev.off()
	}

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	for(HRid in 1:20){

		Result<-HRstore[,HRid]
		Likelihood<-HRstore[,21]

		HRvalues_credints[,HRid]<-GetCredInt(Result,Likelihood,creds,var="",plotgraph=FALSE)
		#dev.off()
	}

	#-------------------------------------------------------

	#Now convert the absolute values of HR into relative values - ie relative to the TARGET HRs

	HRvalues_credints_relative<-HRvalues_credints
	HRvalues_BEST_relative<-HRvalues_BEST/fleet_vector_target
	for(qqq in 1:5){
		HRvalues_credints_relative[qqq,]<-HRvalues_credints[qqq,]/fleet_vector_target
	}

	#-------------------------------------------------------

	#Merge the best fit and credint data into a single data structure for each of gearmults and HR values for outpujt to csv

	gearmult_results<-array(dim=c(6,12),0)
	HRrelative_results<-array(dim=c(6,20),0)
   	colnames(gearmult_results)<- colnames(gearmult_credints)
   	colnames(HRrelative_results)<-colnames(HRvalues_credints)
   	rownames(gearmult_results)<- c("Best",as.character(creds[1:2]),"Median",as.character(creds[4:5]))
   	rownames(HRrelative_results)<- c("Best",as.character(creds[1:2]),"Median",as.character(creds[4:5]))

	   gearmult_results[1,1:12]<-as.numeric(gearmult_BEST[1:12])
   	gearmult_results[2:6,1:12]<-(gearmult_credints)


   	HRrelative_results[1,1:20]<-as.numeric(HRvalues_BEST_relative[1:20])
   	HRrelative_results[2:6,1:20]<-HRvalues_credints_relative

	#Write these two arrays out as csv files
	filename <- csvname(resultsdir, "fitting_to_knownHR_gearmult_stats", model.ident)
	writecsv(gearmult_results, filename)

	filename <- csvname(resultsdir, "fitting_to_knownHR_HR_rel_target_stats", model.ident)
	writecsv(HRrelative_results, filename)


	# Extract the best fit set iof gear mults into a model parameter file structure and export to the /Parameters folder
	extract_ACTmult_to_parmsfolder(model, gearmult_BEST)

	#-------------------------------------------------------
	#-------------------------------------------------------
	#-------------------------------------------------------
	#-------------------------------------------------------
	#-------------------------------------------------------
	#-------------------------------------------------------
	#-------------------------------------------------------


	#NOW SOME DIAGNOSTIC PLOTTING 

	#First thing to do is to disaggregated the ***_results arrays back into
	#a BEST component and the CREDITS component. These already exist in this script
	#but this step is icluded here to facilitate carving the plotting script out
	#as a separate function at some point inthe future maybe. In that case, the two arrays will exist in memory
	#as a list object generated by the annealing code.

	#................................


	#Set up an array with 5 rows to hold the BEST data - the data in each row is going to the the same
	#this is just a trick to make plotting a bit easier later
	gearmult_BEST_array<-array(dim=c(5,12),0)
	HRvalues_BEST_array_relative<-array(dim=c(5,20),0)
	gear_code_list<-colnames(gearmult_results)
	guild_list<-colnames(HRrelative_results)

	colnames(gearmult_BEST_array)<-gear_code_list 
	colnames(HRvalues_BEST_array_relative)<-guild_list

	for(qqq in 1:5){
		gearmult_BEST_array[qqq,]<-gearmult_results[1,]
	}

	for(qqq in 1:5){
		HRvalues_BEST_array_relative[qqq,]<-HRrelative_results[1,]
	}

	#Now extract the credint data into separate arrays too

	gearmult_credints <- gearmult_results[2:6,]
	HRvalues_credints_relative <- HRrelative_results[2:6,]

	#Set up the list objects needed by the bxp function which draws box and whisker plots

	bxpdata_G<-list(stats=gearmult_credints,conf=NULL,out=numeric(length=0),names=colnames(gearmult_credints))
	bxpdata_Gbest<-list(stats=gearmult_BEST_array,conf=NULL,out=numeric(length=0),names=colnames(gearmult_credints))


	bxpdata_H<-list(stats=HRvalues_credints_relative,conf=NULL,out=numeric(length=0),names=colnames(HRvalues_credints))
	bxpdata_Hbest<-list(stats=HRvalues_BEST_array_relative,conf=NULL,out=numeric(length=0),names=colnames(HRvalues_credints))

	#-------------------------------------------------------

	#Plot the gear multipler results 


	par(mfrow=c(1,2))

	par(mar=c(3,4,1,1))
  	bxp(bxpdata_G,boxwex=0.35,at=seq(0,11),horizontal=TRUE,show.names=FALSE,las=1,ylim=c(0,5),boxcol="black",whiskcol="black",whisklty="solid",medcol="black",staplecol="black")
  	bxp(bxpdata_Gbest,boxwex=0.35,add=TRUE,at=seq(0,11),horizontal=TRUE,show.names=FALSE,las=1,boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
	axis(colnames(gearmult_credints),side=2,las=1,at=seq(0,11),cex.axis=1)
	mtext("Gear activity multiplier",cex=1.1,side=1,line=2)
	abline(v=1,lty="dashed")

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


	#Plot the Harvest ratio results 

	par(mar=c(3,4,1,1))
  	bxp(bxpdata_H,boxwex=0.35,at=seq(0,19),horizontal=TRUE,show.names=FALSE,las=1,ylim=c(0,3),boxcol="black",whiskcol="black",whisklty="solid",medcol="black",staplecol="black")
  	bxp(bxpdata_Hbest,boxwex=0.35,add=TRUE,at=seq(0,19),horizontal=TRUE,show.names=FALSE,las=1,boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")
	axis(colnames(HRvalues_credints),side=2,las=1,at=seq(0,19),cex.axis=1)
	mtext("Harvest ratios relative to target",cex=1.1,side=1,line=2)
	abline(v=1,lty="dashed")


	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


}

