#
# annealing_find_gear_activity_mult_ecosystem_target.R
#
#' Optimise StrathE2E fishing gear activity multipliers to maximum the likelihood of observed ecosystem target data
#'
#' Launches a simulated annealing process to find the set of fishing fleet model gear activity multipliers
#' producing the maximum likelihood of observed target data on the state of the ecosystem,
#' given specified environmental driving data and ecology model parameters, and effort-harvest ratio scaling parameters.
#' Note that there is a related function annealing_find_gear_activity_mult_HR_target() which uses only the fishing fleet model to
#' find the activity multipliers which maximise the likelihood of a set of target harvest ratios (where these are known), given values of the
#' scaling values linking effort to harvest ratio.
#'
#' Simulated annealing is an iterative random-walk type process which searches the parameter space
#' of a model to locate the combination which maximises the likelihood of a set of observed
#' data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
#' may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
#' avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
#' by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1)
#'
#' Model configuration and initial values of the ecology model parameters need to be
#' assembled by a prior call of the read_model() function.
#' 
#' NOTE that the user.path argument in the read_model() function call needs to point to a user workspace folder, not the default
#' North Sea model provided with the package. This is because the annealing function needs write-access to the model /Parameters folder,
#' but the /extdata/Models folder in the package installation is read-only.
#' To use the annealing function on the North Sea model, use the copy_model() function to make a copy of the
#' North Sea model in the user workspace.
#'
#' The coefficient of variation for jiggling the activity multipliers can be varied in real-time
#' during the run by editing the file "annealing_SD_fishing.csv" in the folder
#' /Parameters/Parameter_SD_control/ of the model version. Suggested vaues for the SD are in the range 0.1 to 0.01
#'
#' The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
#' the likelihoods of the proposed and accepted parameter sets at each iteration.
#' Y-axis (likelihood of the target data) range of the real time plot can be varied during the run by 
#' editing the setup file "annealing_SD_fishing.csv"
#'
#' At the end of the procedure a new version of the gear activity multipliers file is exported
#' to the folder /Parameters of the model version, with a user defined identifier specified by the model.ident argument
#' in the read_model() function. The histories of proposed and accepted parameter combinations
#' are saved as CSV files in the results folder.
#'
#' To preserve the new activity multipliers and incorporate them into the fishing fleet model parameterisation
#' the multiplier values need to be applied to the activity rates specified in the data input file /Parameters/fishing_activity_parameters_*.csv.
#' Manually update the values in fishing_activity_parameters_*.csv, by multiplying the existing values by the new multipliers emerging from the annealing process.
#'
#' If the edited file fishing_activity_parameters_*.csv is saved with a new identifier (*) then in order to use it in a subsequent run of the
#' StrathE2E model (using the StrathE2E() function) it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new file.
#'
#' @param model R-list object generated by the read_model() function which defined the model configuration
#' @param nyears Number of years to run the model in each iteration (default=40)
#' @param n_iter Number of iterations of the model (default=500)
#' @param start_temperature Initial value of the simulated annealing temperature parameter (default=1). Suggested values in the range 0.0005 - 5. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood
#' @param cooling Rate at which the simulated annealing temperature declines with iterations (default=0.975). Suggested values in the range 0.9 - 0.985
#' @param quiet (TRUE or FALSE) Suppress informational messages at the start of each iteration (default=TRUE)
#'
#' @return CSV file of fitted gear activity multipliers and the histories of the proposed and accepted parameter values
#'
#' @seealso \code{\link{list_models}} , \code{\link{read_model}} , \code{\link{StrathE2E}} , \code{\link{annealing_find_ecology_parameters}} , \code{\link{annealing_find_harvest_ratio_mult}} , \code{\link{annealing_find_gear_activity_mult_HR_target}}
#'
#' @importFrom stats runif
#'
#' @export
#'
#' @examples
#' #Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#' copy_model("North_Sea", "2003-2013",
#'            dest.path="C:/Users/username/Documents/Models")
#'
#' # Load the 2003-2013 version of the North Sea model from the user workspace:
#' model<-read_model(model.name="North_Sea",
#'                   model.variant="2003-2013",
#'                   model.ident="TEST",
#'                   user.path="C:/Users/username/Documents/Models")
#'
#' # Quick Demo of the annealing function in operation:
#' annealing_find_gear_activity_mult_ecosystem_target(model, nyears=5, n_iter=10, start_temperature=0.5)
#' # More realistic configuration would be (WARNING - this will take about 26 hours to run) :
#' annealing_find_gear_activity_mult_ecosystem_target(model, nyears=50, n_iter=1000, start_temperature=1)
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

annealing_find_gear_activity_mult_ecosystem_target <- function(model, nyears=40, n_iter=500, start_temperature=1, cooling=0.975, quiet=TRUE) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	setup		<- elt(model, "setup")
	read.only	<- elt(setup, "read.only")
	model.path	<- elt(setup, "model.path")
	resultsdir	<- elt(setup, "resultsdir")
	identifier	<- elt(setup, "model.ident")

	data				<- elt(model, "data")
	fleet.model			<- elt(data, "fleet.model")
	fitted.parameters		<- elt(data, "fitted.parameters")
	gear_mult			<- elt(fleet.model, "gear_mult")
	HRscale_vector_multiplier	<- elt(fleet.model, "HRscale_vector_multiplier")

	if (read.only) {
		cat("Warning: cannot write fitted parameters back to the model input folders - model is read-only\n")
		cat("Warning: to fix this, make a copy of the model using copy_model() into your own workspace.\n")
		stop("Model is not writable!")
	}

	print(date())


	if (length(which(HRscale_vector_multiplier<1 | HRscale_vector_multiplier>1)) > 0) {
		print("**************************************************************************")
		print("WARNING - one or more baseline Harvest Ratio scaling values differs from 1")
		print("**************************************************************************")
	}

	#Set up a dataframe to hold the proposed and accepted ACT_scale_vector

	ACTmult_proposal<-data.frame(G1=0,G2=0,G3=0,G4=0,G5=0,G6=0,G7=0,G8=0,G9=0,G10=0,G11=0,G12=0,lik=0)
	ACTmult_proposal[1,]<-c(gear_mult,1e-60)
	ACTmult_accepted<-ACTmult_proposal

	n_acceptances <- 0	# Counter for the number of parameter acceptances which have occurred
	lastbest <- 1e-60
	
	temperature <- start_temperature

	#ITERATE THE ANNEALING PROCESS.....

                for (kkk in 1:n_iter){

		#FIRST RUN THROUGH THE MODEL WITH THE CALIBRATION PERIOD HARVEST RATES.....
		#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		annealing_control_data <- read_SD_fishing(model.path)
		axmin <- elt(annealing_control_data, "axmin")
		axmax <- elt(annealing_control_data, "axmax")
		deltaH <- elt(annealing_control_data, "deltaH")		# sd for jiggling the fishing fleet parameters
		#Values for the sd are held in the "annealing_SD_fishing" file.
		#THESE CAN BE ALTERED DURING A RUN BY MANUALLY EDITING THE VAKUES IN THE CONTROL FILE
		# Suggested values are:
		#    deltaH<-0.10  # high value for the first cycle through all this
		#    deltaH<-0.05  # medium value is strating from something half sensible
		#    deltaH<-0.02  # small vakue for the second cycle through all this
		#    deltaH<-0.01  # very small value if strating from something close

		if(kkk>1){
			for(www in 1:12){
				gear_mult[www] <- max(0,rnorm(1,ACTmult_accepted[(kkk-1),www],deltaH*ACTmult_accepted[(kkk-1),www]))
				#Might be useful in some cases: force the MF HR scaler to be same as PF HR scaler
				#if(www==3) HRscale_vector_multiplier[3]<-HRscale_vector_multiplier[1]
			}
			ACTmult_proposal[kkk,]<-c(gear_mult,0)
			ACTmult_accepted[kkk,]<-ACTmult_accepted[(kkk-1),]

			#Now simply drop the new fleet vector into the existing parms vector
			model$data$fleet.model$gear_mult <- gear_mult
		}

		# showall("gear_mult", gear_mult)

		# Build and run the model:
		build <- build_model(model, nyears)
		fit <- StrathE2E_fit(model, build, quiet)
		annual_obj <- elt(fit, "annual_obj")

		# cat("Err=", annual_obj, "\n")

		ACTmult_proposal[kkk,13]<-annual_obj

		temperature<-temperature*cooling

		#-------------------------------------------------------------------
		#Now the Metropolis algorithm.....

		lik_ratio<-exp(((log(annual_obj)) - log(lastbest))/temperature)
		rand<-runif(1,0,1)

		new_accepted<-" Accept_newP:NO"
		if(lik_ratio>rand){
			n_acceptances<-n_acceptances+1
			ACTmult_accepted[kkk,]<-ACTmult_proposal[(kkk),]
			lastbest<-annual_obj

			new_accepted<-" Accept_newP:YES"
		}

		#--------------------------------

		filename = csvname(resultsdir, "annealing_ACTmult_proposalhistory", identifier)
		writecsv(ACTmult_proposal, filename, row.names=FALSE)

		filename = csvname(resultsdir, "annealing_ACTmult_acceptedhistory", identifier)
		writecsv(ACTmult_accepted, filename, row.names=FALSE)

		#-------------------------------------------------------------------

		print(paste("Itn:",kkk," ",ACTmult_proposal[kkk,13]," ",new_accepted,sep=""))
		#print(ACTmult_proposal[kkk,1:12])

		#-------------------------------------------------------------------

		#Plot or update the time series of proposal and acepted likelihoods so far....
		if(kkk>1){
			par(mfrow=c(1,1))

			#Plot the basemodel results 
			plot(seq(1,nrow(ACTmult_proposal)),ACTmult_proposal[,13],ylim=c(axmin,axmax),xlim=c(1,kkk),xlab="Iterations",ylab="Target data likelihood",type="l",col="grey")
			points(seq(1,nrow(ACTmult_accepted)),ACTmult_accepted[,13],type="l",col="black",lwd=3)
                	legend("topleft",c("accepted","proposed"),bg="transparent",col=c("black","grey"),lty=c(1,1),lwd=c(3,1),pt.cex=c(1,1))
		}

	}

	extract_ACTmult_to_parmsfolder(model, ACTmult_accepted)
}

