#
# copy_model.R
#
#' Make a copy of a named model/variant and documentation in a user defined workspace
#'
#' @param model.name	Name of model to copy
#' @param model.variant	Name of model variant to copy
#' @param dest.path	Destination folder to write a copy of model, which will be created if necessary. The default is "Models" in current folder
#' @param user.path	Path to users top level model folder if copying a user model
#
#' @return A copy of an entire model/variant and the associated documentation folder in the designated workspace
#'
#' @seealso \code{\link{list_models}}, \code{\link{read_model}}
#'
#' @export
#'
#' @examples
#' #Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#' copy_model("North_Sea", "2003-2013",
#'            dest.path="C:/Users/username/Documents/Models")
#'
#' #Copy a user model into a user workspace  (Windows OS):
#' copy_model("Modelname", "Modelvariant",
#'            dest.path="C:/Users/username/Documents/Folder2/Models",
#'            user.path="C:/Users/username/Documents/Folder1/Models")
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Ian Thurlbeck                                            |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

copy_model <- function(model.name, model.variant, dest.path="Models", user.path="") {

	src.path <- get.variant.path(model.name, model.variant, user.path)	# path to model variant folder, either system or user folder

	dst.path <- makepath(dest.path, model.name, model.variant)
	if (dir.exists(dst.path)) {
		stop("destination folder '", dst.path, "' already exists !\n", sep="")
	}

	# create top-level model folder:
	dst.path <- makepath(dest.path, model.name)		
	create.folder(dst.path)

	cat(" Copying model: '", model.name, ", variant '", model.variant, "':\n", sep="")
	cat("  from '", src.path, "' to '", dst.path, "'\n", sep="")
	file.copy(src.path, dst.path, recursive=TRUE)

	doc.path <- makepath(dest.path, model.name, "Documentation")
	if (dir.exists(doc.path)) {
		cat("documentation folder '", doc.path, "' already exists !\n", sep="")
	} else {
	src.doc.path <- get.documentation.path(model.name, model.documentation="Documentation", user.path)	# path to model documentation folder, either system or user folder
        file.copy(src.doc.path, dst.path, recursive=TRUE)
	}

	invisible(NULL)
}

