#
# list_models.R
#
#' List the available models in a designated workspace
#'
#' @param path Path to users model folder, otherwise read the list of models supplied with the package.
#'
#' @return An on-screen list of available models in the designated folder
#'
#' @seealso \code{\link{copy_model}}, \code{\link{read_model}}
#'
#' @export
#'
#' @examples
#' #List the models/variants supplied with the package:
#' list_models()
#'
#' #List the models/variants in a user defined workspace:
#' list_models("C:/Users/username/Documents/Folder2/Models")
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Ian Thurlbeck                                            |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

list_models <- function(path="") {

	path <- remove.dirsep(path)	# remove any trailing '/'

	if (path == "") {
		# get path to internal model folder
		full.path <- system.file("extdata/Models", package="StrathE2E2")
	}
	else {
		full.path <- path
	}

	# build up list of models:
	models <- list()
	for (model.dir in list.files(full.path)) {
		model.path <- makepath(full.path, model.dir)				# path to model dir
		if (dir.exists(model.path)) {
			variants <- list()
			for (variant.dir in list.files(model.path)) {
				variant.path <- makepath(model.path, variant.dir)	# path to variant dir
				if (dir.exists(variant.path)) {
					# exists, but check for model setup file:
					setup.file <- makepath(variant.path, MODEL_SETUP)
					if (file.exists(setup.file)) {
						# model/variant/setup.csv exist, so treat this as a model
						variants <- c(variants, variant.dir)
					}
				}
			}
			if (length(variants)) {
				models[[model.dir]] <- variants
			}
		}
	}

	if (length(models)) {
		if (path == "") {
			cat("List of models in system folder")
			user.path <- ""
		} else {
			cat("List of models in user folder")
			user.path <- paste0(", user.path=\"", path, "\"")
		}
		cat(", with helpful commands to load them:\n\n")
		for (model in names(models)) {
			cat(" Model: \"", model, "\"\n", sep="")
			for (variant in models[[model]]) {
				cat ("  Variant: \"", variant, "\"", sep="")
				cat("\tmodel <- read_model(\"", model, "\", \"", variant, "\"", user.path, ")\n", sep="")
			}
			cat("\n")
		}
	}
	else {
		cat("Error: could not find any models in path '", full.path, "' !\n", sep="")
	}
}

