#
# plot_final_year_LTL_inshore_vs_offshore.R
#
#' Plots of inshore and offshore low trophic level state variables for the final year of a model run
#'
#' Multi-panel time series plots of inshore and offshore low trophic level state variables for the final year of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series plots of daily values of nutrients, detritus, macrophyte, phytoplankton, zooplankton and suspension/deposit feeding benthos (including larvae) for the inshore zone and upper and lower levels of the offshore zone, for the final year of a model run.
#'
#' Units of the plotted varaiables are mass concentrations, scaled to seas surface area or layer volume, ie. mMN/m2 or mMN/m3
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_final_year}} , \code{\link{plot_final_year_HTL_inshore_vs_offshore}} , \code{\link{plot_sediment_habitats_final_year}}
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the final year of low trophic level outputs
#' plot_final_year_LTL_inshore_vs_offshore(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_final_year_LTL_inshore_vs_offshore(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_final_year_LTL_inshore_vs_offshore(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the final year low trophic level output
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_final_year_LTL_inshore_vs_offshore(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_final_year_LTL_inshore_vs_offshore(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------




plot_final_year_LTL_inshore_vs_offshore <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	build		<- elt(results, "build")
	run		<- elt(build, "run")
	nyears		<- elt(run, "nyears")
	ndays		<- elt(run, "ndays")

	data		<- elt(model, "data")
	physical.parms	<- elt(data, "physical.parameters")
	si_depth	<- elt(physical.parms, "si_depth")
	so_depth	<- elt(physical.parms, "so_depth")
	d_depth		<- elt(physical.parms, "d_depth")
	x_shallowprop	<- elt(physical.parms, "x_shallowprop")

	output		<- elt(results, "output")
	corpse_d1	<- elt(output, "corpse_d1")
	corpse_d2	<- elt(output, "corpse_d2")
	corpse_d3	<- elt(output, "corpse_d3")
	corpse_s1	<- elt(output, "corpse_s1")
	corpse_s2	<- elt(output, "corpse_s2")
	corpse_s3	<- elt(output, "corpse_s3")
	kelpdebris	<- elt(output, "kelpdebris")
	detritus_so	<- elt(output, "detritus_so")
	detritus_si	<- elt(output, "detritus_si")
	detritus_d	<- elt(output, "detritus_d")
	nitrate_so	<- elt(output, "nitrate_so")
	nitrate_si	<- elt(output, "nitrate_si")
	nitrate_d	<- elt(output, "nitrate_d")
	ammonia_so	<- elt(output, "ammonia_so")
	ammonia_si	<- elt(output, "ammonia_si")
	ammonia_d	<- elt(output, "ammonia_d")
	phyt_so		<- elt(output, "phyt_so")
	phyt_si		<- elt(output, "phyt_si")
	phyt_d		<- elt(output, "phyt_d")
	herb_o		<- elt(output, "herb_o")
	herb_i		<- elt(output, "herb_i")
	carn_o		<- elt(output, "carn_o")
	carn_i		<- elt(output, "carn_i")
	benths_o	<- elt(output, "benths_o")
	benths_i	<- elt(output, "benths_i")
	kelpN		<- elt(output, "kelpN")
	benthslar_o	<- elt(output, "benthslar_o")
	benthslar_i	<- elt(output, "benthslar_i")


	aggregates	<- elt(results, "aggregates")
	x_poros		<- elt(aggregates, "x_poros")
	x_depth		<- elt(aggregates, "x_depth")

	xvolume_si<-si_depth*x_shallowprop
	xvolume_so<-so_depth*(1-x_shallowprop)
	xd_volume<-d_depth*(1-x_shallowprop)

	par(mfrow=c(3,3))

	l1<-(corpse_d1[((nyears-1)*360+1):ndays]+corpse_d2[((nyears-1)*360+1):ndays]+corpse_d3[((nyears-1)*360+1):ndays])/(1-x_shallowprop)
	l2<-(corpse_s1[((nyears-1)*360+1):ndays]+corpse_s2[((nyears-1)*360+1):ndays]+corpse_s3[((nyears-1)*360+1):ndays])/x_shallowprop
	l3<-(kelpdebris[((nyears-1)*360+1):ndays])/x_shallowprop
#	tsplot33("Corpse&Macrophyt. debris","Nitrogen/m2","Offshore","Inshore","Macrop. debris",l1,l2,l3)
	fyplot3("Corpse&Macrophyt. debris","Nitrogen/m2","Offshore","Inshore","Macrop. debris",l1,l2,l3)

	l1<-detritus_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-detritus_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-detritus_d[((nyears-1)*360+1):ndays]/xd_volume
	#l3<-1000*100*(((x_detritus[((nyears-1)*360+1):ndays])*14)/1000)/(x_depth*(((1-x_poros)*(2650*1000))))
	#This converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)
	#Then scale by 1000 to get on same axes as water detritus mMN/m3 water
#	tsplot33("Detritus","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)
	fyplot3("Detritus","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)

	l1<-nitrate_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-nitrate_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-nitrate_d[((nyears-1)*360+1):ndays]/xd_volume
	#l3<-x_nitrate[((nyears-1)*360+1):ndays]/(x_depth*x_poros)
	#This converts the sediment nitrate into units of N /m3 in the pore water)
#	tsplot33("Nitrate","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)
	fyplot3("Nitrate","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)

	l1<-ammonia_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-ammonia_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-ammonia_d[((nyears-1)*360+1):ndays]/xd_volume
	#l3<-(x_ammonia[((nyears-1)*360+1):ndays]/(x_depth*x_poros))/10
	#This converts the sediment nitrate into units of N /m3 in the pore water)
#	tsplot33("Ammonia","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)
	fyplot3("Ammonia","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)

	l1<-phyt_so[((nyears-1)*360+1):ndays]/xvolume_so
	l2<-phyt_si[((nyears-1)*360+1):ndays]/xvolume_si
	l3<-phyt_d[((nyears-1)*360+1):ndays]/xd_volume
#	tsplot33("Phytoplankton","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)
	fyplot3("Phytoplankton","Nitrogen/m3","S-offshore","S-inshore","Deep",l1,l2,l3)

	l1<-herb_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-herb_i[((nyears-1)*360+1):ndays]/xvolume_si
#	tsplot22("Omniv. zooplankton","Nitrogen/m3","Offshore","Inshore",l1,l2)
	fyplot2("Omniv. zooplankton","Nitrogen/m3","Offshore","Inshore",l1,l2)

	l1<-carn_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-carn_i[((nyears-1)*360+1):ndays]/xvolume_si
#	tsplot22("Carn. zooplankton","Nitrogen/m3","Offshore","Inshore",l1,l2)
	fyplot2("Carn. zooplankton","Nitrogen/m3","Offshore","Inshore",l1,l2)

	l1<-benths_o[((nyears-1)*360+1):ndays]/(1-x_shallowprop)
	l2<-benths_i[((nyears-1)*360+1):ndays]/x_shallowprop
	l3<-kelpN[((nyears-1)*360+1):ndays]/x_shallowprop
#	tsplot33("Macrophyte & Benthos s/d","Nitrogen/m2","OffshoreB","InshoreB","Macrop.",l1,l2,l3)
	fyplot3("Macrophyte & Benthos s/d","Nitrogen/m2","OffshoreB","InshoreB","Macrop.",l1,l2,l3)

	l1<-benthslar_o[((nyears-1)*360+1):ndays]/(xvolume_so+xd_volume)
	l2<-benthslar_i[((nyears-1)*360+1):ndays]/xvolume_si
#	tsplot22("Benthos s/d larvae","Nitrogen/m3","Offshore","Inshore",l1,l2)
	fyplot2("Benthos s/d larvae","Nitrogen/m3","Offshore","Inshore",l1,l2)

}

