#
# plot_for_each_guild_catch_by_gear.R
#
#' Plots of the distributions of annual integrated catches of each guild across gears for the final year of a model run
#'
#' Multi-panel bar-plots of the distributions of annual integrated catches of each guild across gears for the final year of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of bar-plots of annual integrated catches
#' of the guilds in the ecology model which are potentially catchable by the gears in the fishing fleet model:
#' macrophytes, carnivorous zooplankton, suspension/deposit and carnivore/scavenge feeding benthos, planktivorous, demersal and migratory fish,
#' birds, pinnipeds and cetaceans.
#'
#' Each panel represesents a single guild. In each panel, the x-axis represents the (up to) 12 gear types parameterised in the fishing fleet model, and the y-axis is the annual integrated catch.
#' For each gear there are two bars corresponding to the inshore and offshore zones. Each bar is subdivided into landed and discarded components of the catch.
#' Green and blue sections are offshore and inshore landings respectively; black and grey sections are offshore and inshore discards respectively.
#'
#' Units for the catch are mMN/m2/year
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_time_series_annual_land_disc}} , \code{\link{plot_for_each_gear_catch_by_guild}} 
#'
#' @importFrom graphics barplot title
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the time series of annual landings and discards
#' plot_for_each_guild_catch_by_gear(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_for_each_guild_catch_by_gear(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_for_each_guild_catch_by_gear(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the distribution of catches across gears
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_for_each_guild_catch_by_gear(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model$data$fleet.model$gear_mult[1] <- 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_for_each_guild_catch_by_gear(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------


plot_for_each_guild_catch_by_gear <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	data			<- elt(model, "data")
	fleet.model		<- elt(data, "fleet.model")
	gear_codes		<- elt(fleet.model, "gear_codes")
	
	final.year.outputs	<- elt(results, "final.year.outputs")
	offshore_catchmat	<- elt(final.year.outputs, "offshore_catchmat")
	offshore_discmat	<- elt(final.year.outputs, "offshore_discmat")
	offshore_landmat	<- elt(final.year.outputs, "offshore_landmat")

	inshore_catchmat	<- elt(final.year.outputs, "inshore_catchmat")
	inshore_discmat		<- elt(final.year.outputs, "inshore_discmat")
	inshore_landmat		<- elt(final.year.outputs, "inshore_landmat")

	#----------------------------------------------------------------
	#In each plot panel:
	#    black and grey are offshore and inshore discards respectively
	#    green and blue are offshore and inshore landings respectively
	#----------------------------------------------------------------
	ngroups <- 10

	par(mfrow=c(6,2))
	par(mar=c(4,5,1,0.5))

	for(dsa in 1:(ngroups+1)) {

		mt<-(rownames(offshore_catchmat))[dsa]

		offshore_data2plot<-rbind(offshore_discmat[dsa,],offshore_landmat[dsa,])
		inshore_data2plot<-rbind(inshore_discmat[dsa,],inshore_landmat[dsa,])

		#colnames(offshore_data2plot)<-c("PT","PS","LLm","BT","DS","OT","GN","ST","NT","CR","MD","--")
		colnames(offshore_data2plot)<-gear_codes

		if(sum(offshore_data2plot+inshore_data2plot)>0) {
			yaxmax<-(1.2*max(c(offshore_catchmat[dsa,],inshore_catchmat[dsa,])))
		}

		if(sum(offshore_data2plot+inshore_data2plot)==0) {
			yaxmax<-1
		}

		barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,yaxmax),xlim=c(0,12),width=rep(0.4,12), space=c(0.5,rep(1.2,11)),yaxt="n",ann=FALSE,cex.axis=0.6)
		axis(side=2,las=1,cex.axis=0.9)
		if(dsa==1)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==3)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==5)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==7)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==9)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==11)  mtext("Catch",cex=0.7,side=2,line=3.5)
		title(main=mt,cex.main=1.0)
		barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,12), space=c(1.5,rep(1.2,11)),yaxt="n",xaxt="n",ann=FALSE)

	}

	#Calculate total catch aand discard by gear
	mt<-"All guilds combined"

	offshore_data2plot<-rbind(colSums(offshore_discmat),colSums(offshore_landmat))
	inshore_data2plot<-rbind(colSums(inshore_discmat),colSums(inshore_landmat))
	colnames(offshore_data2plot)<-gear_codes

	if(sum(offshore_data2plot+inshore_data2plot)>0) {
		yaxmax<-(1.2*max(c(colSums(offshore_catchmat),colSums(inshore_catchmat))))
	}

	if(sum(offshore_data2plot+inshore_data2plot)==0) {
		yaxmax<-1
	}

	barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,yaxmax),xlim=c(0,12),width=rep(0.4,12), space=c(0.5,rep(1.2,11)),yaxt="n",ann=FALSE,cex.axis=0.5)
	axis(side=2,las=1,cex.axis=0.9)
	if(dsa==1)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==3)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==5)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==7)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==9)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==11)  mtext("Catch",cex=0.7,side=2,line=3.5)
	title(main=mt,cex.main=1.0)
	barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,12), space=c(1.5,rep(1.2,11)),yaxt="n",xaxt="n",ann=FALSE)

}

