#
# plot_sediment_habitats_final_year.R
#
#' Plots of inshore and offshore sediment habitat state variables
#'
#' Multi-panel time series plots of inshore and offshore sediment habitat state variables for the final year of a model run generated by the StrathE2E() function
#'
#' The function plots a multi-panel page of time series plots of daily values of nutrients and detritus for the various seabed sediment classes in the inshore and offshore zones, for the final year of a model run.
#'
#' Units of the plotted varaiables are mass concentrations, scaled to sea surface area or layer volume, ie. mMN/m2 or mMN/m3,
#' except for sediment organic nitrogen content which is expressed as percentage of sediment dry weight to conform with the convention for presenting observational data.
#'
#' @param model R-list object defining the model configuration compiled by the read_model() function
#' @param results R-list object containing model results generated by the StrathE2E() function
#'
#' @return Graphical display in a new graphics window
#'
#' @seealso \code{\link{read_model}}, \code{\link{StrathE2E}}, \code{\link{plot_final_year}} , \code{\link{plot_final_year_LTL_inshore_vs_offshore}} , \code{\link{plot_final_year_HTL_inshore_vs_offshore}}
#'
#' @export
#'
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package:
#' model <- read_model("North_Sea", "2003-2013")
#' #Run the model and generate the results object
#' results <- StrathE2E(model,nyears=10)
#' # Plot the final year of low trophic level outputs
#' plot_sediment_habitats_final_year(model, results)
#'
#' # Time series plot of state varaiables in a new window leaving any existing windows open
#' dev.new()
#' plot_sediment_habitats_final_year(model, results)
#'
#' #Direct the graphics output to a file (Windows OS)... 
#' pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=8,height=6)      # or jpeg("plot.jpg"), png("plot.png")
#' 	plot_sediment_habitats_final_year(model, results)
#' dev.off()
#'
#' # Read the North Sea/1970-1999 model and set the identifier for output files to "Baseline", run the model and plot the final year sediment haabitat output
#' model <- read_model("North_Sea", "1970-1999",model.ident="Baseline")
#' results <- StrathE2E(model,nyears=10)
#' plot_sediment_habitats_final_year(model, results)
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gears 4,5,6,8 and 9 (all sea-bed contact gears) by a factor of 2
#' model$data$fleet.model$gear_mult[c(4,5,6,8,9)] <- 2
#' model$setup$model.ident <- "bottomgearsx2"        # Set a new identifier for the outputs
#' results <- StrathE2E(model,nyears=10)
#' dev.new()                                   # Open a new graphics window   
#' plot_sediment_habitats_final_year(model, results)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

plot_sediment_habitats_final_year <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	data			<- elt(model, "data")
	physical.parameters	<- elt(data, "physical.parameters")
	x_poros_s1		<- elt(physical.parameters, "x_poros_s1")
	x_poros_s2		<- elt(physical.parameters, "x_poros_s2")
	x_poros_s3		<- elt(physical.parameters, "x_poros_s3")
	x_area_s1		<- elt(physical.parameters, "x_area_s1")
	x_area_s2		<- elt(physical.parameters, "x_area_s2")
	x_area_s3		<- elt(physical.parameters, "x_area_s3")
	x_depth_s1		<- elt(physical.parameters, "x_depth_s1")
	x_depth_s2		<- elt(physical.parameters, "x_depth_s2")
	x_depth_s3		<- elt(physical.parameters, "x_depth_s3")

	x_poros_d1		<- elt(physical.parameters, "x_poros_d1")
	x_poros_d2		<- elt(physical.parameters, "x_poros_d2")
	x_poros_d3		<- elt(physical.parameters, "x_poros_d3")
	x_area_d1		<- elt(physical.parameters, "x_area_d1")
	x_area_d2		<- elt(physical.parameters, "x_area_d2")
	x_area_d3		<- elt(physical.parameters, "x_area_d3")
	x_depth_d1		<- elt(physical.parameters, "x_depth_d1")
	x_depth_d2		<- elt(physical.parameters, "x_depth_d2")
	x_depth_d3		<- elt(physical.parameters, "x_depth_d3")

	build			<- elt(results, "build")
	run			<- elt(build, "run")
	nyears			<- elt(run, "nyears")
	ndays			<- elt(run, "ndays")

	output		<- elt(results, "output")
	x_ammonia_s1	<- elt(output, "x_ammonia_s1")
	x_ammonia_s2	<- elt(output, "x_ammonia_s2")
	x_ammonia_s3	<- elt(output, "x_ammonia_s3")
	x_ammonia_d1	<- elt(output, "x_ammonia_d1")
	x_ammonia_d2	<- elt(output, "x_ammonia_d2")
	x_ammonia_d3	<- elt(output, "x_ammonia_d3")
	x_nitrate_s1	<- elt(output, "x_nitrate_s1")
	x_nitrate_s2	<- elt(output, "x_nitrate_s2")
	x_nitrate_s3	<- elt(output, "x_nitrate_s3")
	x_nitrate_d1	<- elt(output, "x_nitrate_d1")
	x_nitrate_d2	<- elt(output, "x_nitrate_d2")
	x_nitrate_d3	<- elt(output, "x_nitrate_d3")
	x_detritus_s1	<- elt(output, "x_detritus_s1")
	xR_detritus_s1	<- elt(output, "xR_detritus_s1")
	x_detritus_s2	<- elt(output, "x_detritus_s2")
	xR_detritus_s2	<- elt(output, "xR_detritus_s2")
	x_detritus_s3	<- elt(output, "x_detritus_s3")
	xR_detritus_s3	<- elt(output, "xR_detritus_s3")
	x_detritus_d1	<- elt(output, "x_detritus_d1")
	xR_detritus_d1	<- elt(output, "xR_detritus_d1")
	x_detritus_d2	<- elt(output, "x_detritus_d2")
	xR_detritus_d2	<- elt(output, "xR_detritus_d2")
	x_detritus_d3	<- elt(output, "x_detritus_d3")
	xR_detritus_d3	<- elt(output, "xR_detritus_d3")
	corpse_s1	<- elt(output, "corpse_s1")
	corpse_s2	<- elt(output, "corpse_s2")
	corpse_s3	<- elt(output, "corpse_s3")
	corpse_d1	<- elt(output, "corpse_d1")
	corpse_d2	<- elt(output, "corpse_d2")
	corpse_d3	<- elt(output, "corpse_d3")

	#Plot the final year of output for the sediment habitats separately

	#Plots are:
	#    Shallow layer - wc ammonia, hab1 pw ammonia, hab2 pw ammonia, hab3 pw ammonia
	#    Deep layer    - wc ammonia, hab1 pw ammonia, hab2 pw ammonia, hab3 pw ammonia

	#    Shallow layer - wc nitrate, hab1 pw nitrate, hab2 pw nitrate, hab3 pw nitrate
	#    Deep layer    - wc nitrate, hab1 pw nitrate, hab2 pw nitrate, hab3 pw nitrate

	#    Shallow layer - wc detritus, hab1 detritus, hab2 detritus, hab3 pw detritus
	#    Deep layer    - wc detritus, hab1 detritus, hab2 detritus, hab3 pw detritus

	par(mfrow=c(4,2))

	#Shallow sediment ammonia

	if(x_poros_s1>0 && x_area_s1>0){
		l1<-x_ammonia_s1[((nyears-1)*360+1):ndays]/(x_area_s1*x_depth_s1*x_poros_s1)
	} else {
		l1<-rep(NA,361)
	}

	if(x_poros_s2>0 && x_area_s2>0){
		l2<-x_ammonia_s2[((nyears-1)*360+1):ndays]/(x_area_s2*x_depth_s2*x_poros_s2)
	} else {
		l2<-rep(NA,361)
	}

	if(x_poros_s3>0 && x_area_s3>0){
		l3<-x_ammonia_s3[((nyears-1)*360+1):ndays]/(x_area_s3*x_depth_s3*x_poros_s3)
	} else {
		l3<-rep(NA,361)
	}

	#This converts the sediment ammonia into units of N /m3 in the pore water)
	fyplot3_hab("Inshore ammonia","Nitrogen/m3","Area_s1 porewater","Area_s2 porewater","Area_s3 porewater",l1,l2,l3)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	#Deep sediment ammonia

	if(x_poros_d1>0 && x_area_d1>0){
		l1<-x_ammonia_d1[((nyears-1)*360+1):ndays]/(x_area_d1*x_depth_d1*x_poros_d1)
	} else {
		l1<-rep(NA,361)
	}

	if(x_poros_d2>0 && x_area_d2>0){
		l2<-x_ammonia_d2[((nyears-1)*360+1):ndays]/(x_area_d2*x_depth_d2*x_poros_d2)
	} else {
		l2<-rep(NA,361)
	}

	if(x_poros_d3>0 && x_area_d3>0){
		l3<-x_ammonia_d3[((nyears-1)*360+1):ndays]/(x_area_d3*x_depth_d3*x_poros_d3)
	} else {
		l3<-rep(NA,361)
	}

	#This converts the sediment ammonia into units of N /m3 in the pore water)
	fyplot3_hab("Offshore ammonia","Nitrogen/m3","Area_d1 porewater","Area_d2 porewater","Area_d3 porewater",l1,l2,l3)

	#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Shallow sediment nitrate

if(x_poros_s1>0 && x_area_s1>0){
l1<-x_nitrate_s1[((nyears-1)*360+1):ndays]/(x_area_s1*x_depth_s1*x_poros_s1)
} else {
l1<-rep(NA,361)
}

if(x_poros_s2>0 && x_area_s2>0){
l2<-x_nitrate_s2[((nyears-1)*360+1):ndays]/(x_area_s2*x_depth_s2*x_poros_s2)
} else {
l3<-rep(NA,361)
}

if(x_poros_s3>0 && x_area_s3>0){
l3<-x_nitrate_s3[((nyears-1)*360+1):ndays]/(x_area_s3*x_depth_s3*x_poros_s3)
} else {
l3<-rep(NA,361)
}

#This converts the sediment nitrate into units of N /m3 in the pore water)
fyplot3_hab("Inshore nitrate","Nitrogen/m3","Area_s1 porewater","Area_s2 porewater","Area_s3 porewater",l1,l2,l3)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Deep sediment nitrate

if(x_poros_d1>0 && x_area_d1>0){
l1<-x_nitrate_d1[((nyears-1)*360+1):ndays]/(x_area_d1*x_depth_d1*x_poros_d1)
} else {
l1<-rep(NA,361)
}

if(x_poros_d2>0 && x_area_d2>0){
l2<-x_nitrate_d2[((nyears-1)*360+1):ndays]/(x_area_d2*x_depth_d2*x_poros_d2)
} else {
l2<-rep(NA,361)
}

if(x_poros_d3>0 && x_area_d3>0){
l3<-x_nitrate_d3[((nyears-1)*360+1):ndays]/(x_area_d3*x_depth_d3*x_poros_d3)
} else {
l3<-rep(NA,361)
}

#This converts the sediment nitrate into units of N /m3 in the pore water)
fyplot3_hab("Offshore nitrate","Nitrogen/m3","Area_d1 porewater","Area_d2 porewater","Area_d3 porewater",l1,l2,l3)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Shallow sediment total detritus

if(x_poros_s1>0 && x_area_s1>0){
l1a<-100*(((x_detritus_s1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s1*x_depth_s1*(((1-x_poros_s1)*(2650*1000))))
l1b<-100*(((xR_detritus_s1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s1*x_depth_s1*(((1-x_poros_s1)*(2650*1000))))
} else {
l1a<-rep(NA,361)
l1b<-rep(NA,361)
}

if(x_poros_s2>0 && x_area_s2>0){
l2a<-100*(((x_detritus_s2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s2*x_depth_s2*(((1-x_poros_s2)*(2650*1000))))
l2b<-100*(((xR_detritus_s2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s2*x_depth_s2*(((1-x_poros_s2)*(2650*1000))))
} else {
l2a<-rep(NA,361)
l2b<-rep(NA,361)
}

if(x_poros_s3>0 && x_area_s3>0){
l3a<-100*(((x_detritus_s3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s3*x_depth_s3*(((1-x_poros_s3)*(2650*1000))))
l3b<-100*(((xR_detritus_s3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_s3*x_depth_s3*(((1-x_poros_s3)*(2650*1000))))
} else {
l3a<-rep(NA,361)
l3b<-rep(NA,361)
}
 
l1<-l1a+l1b
l2<-l2a+l2b
l3<-l3a+l3b
#This converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)
#Then scale by 1000 to get on same axes as water detritus mMN/m3 water
fyplot3_hab("Inshore detritus","Nitrogen/DW (g/g %)","Area_s1 sediment","Area_s2 sediment","Area_s3 sediment",l1,l2,l3)


#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Deep sediment total detritus


if(x_poros_d1>0 && x_area_d1>0){
l1a<-100*(((x_detritus_d1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d1*x_depth_d1*(((1-x_poros_d1)*(2650*1000))))
l1b<-100*(((xR_detritus_d1[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d1*x_depth_d1*(((1-x_poros_d1)*(2650*1000))))
} else {
l1a<-rep(NA,361)
l1b<-rep(NA,361)
}

if(x_poros_d2>0 && x_area_d2>0){
l2a<-100*(((x_detritus_d2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d2*x_depth_d2*(((1-x_poros_d2)*(2650*1000))))
l2b<-100*(((xR_detritus_d2[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d2*x_depth_d2*(((1-x_poros_d2)*(2650*1000))))
} else {
l2a<-rep(NA,361)
l2b<-rep(NA,361)
}

if(x_poros_d3>0 && x_area_d3>0){
l3a<-100*(((x_detritus_d3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d3*x_depth_d3*(((1-x_poros_d3)*(2650*1000))))
l3b<-100*(((xR_detritus_d3[((nyears-1)*360+1):ndays])*14)/1000)/(x_area_d3*x_depth_d3*(((1-x_poros_d3)*(2650*1000))))
} else {
l3a<-rep(NA,361)
l3b<-rep(NA,361)
}

l1<-l1a+l1b
l2<-l2a+l2b
l3<-l3a+l3b
#This converts the sediment detritus into units of %N by dry wt (100*gN/g-drysediment) (density of dry solid matter = 2.65g/cm3)
#Then scale by 1000 to get on same axes as water detritus mMN/m3 water
fyplot3_hab("Offshore detritus","Nitrogen/DW (g/g %)","Area_d1 sediment","Area_d2 sediment","Area_d3 sediment",l1,l2,l3)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Shallow sediment corpses

if(x_area_s1>0){
l1<-corpse_s1[((nyears-1)*360+1):ndays]/(x_area_s1)
} else {
l1<-rep(NA,361)
}

if(x_area_s2>0){
l2<-corpse_s2[((nyears-1)*360+1):ndays]/(x_area_s2)
} else {
l2<-rep(NA,361)
}

if(x_area_s3>0){
l3<-corpse_s3[((nyears-1)*360+1):ndays]/(x_area_s3)
} else {
l3<-rep(NA,361)
}

#This converts the sediment corpse mass into units of N /m2 of sediment surface)
fyplot3_hab("Inshore corpses","Nitrogen/m2","Area_s1","Area_s2","Area_s3",l1,l2,l3)

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#Deep sediment corpses

if(x_area_d1>0){
l1<-corpse_d1[((nyears-1)*360+1):ndays]/(x_area_d1)
} else {
l1<-rep(NA,361)
}

if(x_area_d2>0){
l2<-corpse_d2[((nyears-1)*360+1):ndays]/(x_area_d2)
} else {
l2<-rep(NA,361)
}

if(x_area_d3>0){
l3<-corpse_d3[((nyears-1)*360+1):ndays]/(x_area_d3)
} else {
l3<-rep(NA,361)
}

#This converts the sediment corpse mass into units of N /m2 of sediment surface)
fyplot3_hab("Offshore corpses","Nitrogen/m2","Area_d1","Area_d2","Area_d3",l1,l2,l3)


#-------------------------------------------------------------------------------------------------------


}

