#
# read_model.R
#
#' Read all the configuration, driving and parameter files for a designated model and compiles the data into a list object
#'
#' This function is the key point of entry to the package. It reads all the CSV files containing the configuration, driving and parameter values which define
#' a model for a particular region and compiles them into a single R-list object. This data object forms the basis for almost all other functions in the package.
#'
#' The input CSV files are specified in the MODEL_SETUP.csv file located
#' in the Model/variant folder specified by the argument user.path in a read_model() call, or from the extdata/Models folder
#' in the package installation. The models supplied with the package are two variants (1970-1999 and 2003-2013) of a fully
#' parameterised and documented model of the North Sea.
#'
#' Starting from a baseline model configuration defined in the MODEL_SETUP.csv file, any of the terms in the R-list objcet created by read_model() can be modified by coding
#' to produce an unlimited range of scenario configurations representing, for example, changes in the physical environment, changes in the composition or activity of the
#' fishing fleets, or any combination of these. The power of this approach is that large numbers of scanarios can be coded in standard R and simulated without any need
#' for manual editing of input files.
#'
#' @param model.name Name of model to read (no default)
#' @param model.variant Variant of the model to be read (no default)
#' @param model.ident Text appended to output file names (e.g. OFFSHORE_model_annualresults-TEXT.csv instead of just OFFSHORE_model_annualresults.csv) (default="base")
#' @param model.subdir Sub directory of the main results folder for storing results (default = a folder which will be auto-created using the model name)
#' @param user.path Path to users top level model folder from where the model setup will be loaded (default = load from the package folder extdata/Models)
#' @param quiet Set to FALSE to see on-screen information on where model is loaded from and what files are read (default=TRUE)
#'
#' @return R-list object containing all of the model configuration data
#'
#' @seealso \code{\link{list_models}}, \code{\link{copy_model}}, \code{\link{StrathE2E}}
#'
#' @export
#
#' @examples
#' # Load the 2003-2013 version of the North Sea model supplied with the package and set a value for the identifier to be added to output file names; outputs go to a default results folder:
#' model <- read_model("North_Sea", "2003-2013",model.ident="Demo")
#'
#' # Load a user defined model from a user workspace (Windows OS):
#' model<-read_model("Modelname", "Variantname",
#'                  model.ident="Demo",
#'                  user.path="C:/Users/username/Documents/Foldername/Models")
#'
#' # Load a user defined model from a user workspace and send the model outputs to a specified folder (Windows OS):
#' model<-read_model("Modelname", "Variantname",
#'                  model.ident="Demo",
#'                  model.subdir="C:/Users/username/Documents/Foldername/results",
#'                  user.path="C:/Users/username/Documents/Foldername/Models")
#'
#' # View details of the structure and contents of the model list-object
#' str(model)
#'
#' # Create a new scenario version of the North Sea/1970-1999 model by increasing the activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
#' model <- read_model("North_Sea", "1970-1999",model.ident="Gear1x2")
#' model$data$fleet.model$gear_mult[1] <- 2    # Change the gear activity multiplier for gear 1 to a value of 2
#' model$setup$model.ident <- "gear1x2"        # Set a new identifier for the outputs
#'
#' # Load a baseline case of the North Sea model with 2003-2013 conditions, and then create a scenario with the temperature in all zones increased by 1C
#' baseTemp_model<-read_model("North_Sea", "2003-2013",model.ident="baseTemp")    # Read the embedded North Sea 2003-2013 model and assign an identifier for the results
#' baseTemp_results<-StrathE2E(baseTemp_model,nyears=40)                          # Run the model and save the results to a named list object
#' plot_full_length_timeseries(baseTemp_model,baseTemp_results)                   # Visualise the tull-length time series of output from the run
#' baseTemp_plus1C_model<-baseTemp_model                                          # copy the baseline model list object to a new model list
#' baseTemp_plus1C_model$data$physics.drivers$so_temp <- baseTemp_model$data$physics.drivers$so_temp+1   # add 1 degC to upper layer offshore temperatures
#' baseTemp_plus1C_model$data$physics.drivers$si_temp <- baseTemp_model$data$physics.drivers$si_temp+1   # add 1 degC to inshore temperatures
#' baseTemp_plus1C_model$data$physics.drivers$d_temp  <- baseTemp_model$data$physics.drivers$d_temp+1    # add 1 degC to lower layer offshore temperatures
#' baseTemp_plus1C_model$setup$model.ident <- "baseTemp_plus1C"                   # Assign a unique identifier to each set of .csv outputs
#' baseTemp_plus1C_results<-StrathE2E(baseTemp_plus1C_model,nyears=40)            # Run the model and save the results to a named list object
#' dev.new()                                                                      # Open a new graphics window
#' plot_full_length_timeseries(baseTemp_plus1C_model,baseTemp_plus1C_results)     # Visualise the tull-length time series of output from the run
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: October 2019                                |
# |                                                                   |
# ---------------------------------------------------------------------

read_model <- function(model.name, model.variant, model.ident="base", model.subdir="", user.path="", quiet=TRUE) {

	pkg.env$quiet	<- quiet			# quietens down read/write notes

	read.only	<- (user.path == "")							# read only unless user path is specified

	model.path	<- get.variant.path(model.name, model.variant, user.path)		# full path to either the system model or the user specified one:
	resultsdir	<- makepath(MODEL_RESULTS_DIR, model.name, model.variant, model.subdir)	# results/<model>/<variant>/<subdir>

	setup <- list(
		read.only	= read.only,
		model.name	= model.name,		# "NorthSea"
		model.variant	= model.variant,	# "2000-2013"
		model.ident	= model.ident,		# "speculative"
		model.subdir	= model.subdir,		# "test"
		model.path	= model.path,		# "../Models/NorthSea/2000-2013"
		resultsdir	= resultsdir		# "results/NorthSea/2000-2013/test" ...
	)

	if (! pkg.env$quiet) cat("Loading model   : ", model.path, "\n", sep="")

	read.model.setup(model.path)			# Models/Model/Variant/MODEL_SETUP.csv

	# read model inputs:
	physical.parameters	<- read_physical_parameters(model.path)
	fixed.parameters	<- read_fixed_parameters(model.path)
	physics.drivers		<- read_physics_drivers(model.path)
	chemistry.drivers	<- read_chemistry_drivers(model.path)
	biological.events	<- read_biological_event_timings(model.path)
	fitted.parameters	<- read_fitted_parameters(model.path)
	initial.state		<- read_initial_state(model.path)
	fleet.model		<- read_fishing_fleet_model(model.path, physical.parameters)

	# data slot:
	data <- list(
		# read:
		fixed.parameters	= fixed.parameters,
		fitted.parameters	= fitted.parameters,
		physical.parameters	= physical.parameters,
		physics.drivers		= physics.drivers,
		chemistry.drivers	= chemistry.drivers,
		biological.events	= biological.events,
		fleet.model		= fleet.model,
		initial.state		= initial.state
	)

	model <- list(
		setup		= setup,
		data		= data
	)

	model
}

