## ----global_options, include=FALSE--------------------------------------------
knitr::opts_chunk$set(out.extra = '', fig.pos = 'H', collapse = TRUE, comment = NA)

## ----launchoverview1, eval = FALSE--------------------------------------------
#  library("StrathE2E2")	# Load the package
#  ?StrathE2E2		# Launch the overview document in a browser window
#  # or...
#  help(StrathE2E2)

## ----launchoverview2, eval = TRUE---------------------------------------------
citation("StrathE2E2")

## ----quickstart, fig.width=7, fig.height=5, fig.align='center'----------------
library("StrathE2E2")				# Load the package
model<-read_model("North_Sea", "1970-1999")	# Read the input data for the 1970-1999 version of the North Sea model
results <- StrathE2E(model,nyears=5)		# Run the model for 5 years
plot_full_length_timeseries(model,results)	# Plot time series of the masses of the state variables

## ----loadpackage--------------------------------------------------------------
library("StrathE2E2")

## ----listmodels---------------------------------------------------------------
library("StrathE2E2")
list_models()

## ----copymodels, eval = FALSE-------------------------------------------------
#  library("StrathE2E2")
#  copy_model (" North_Sea ", " mymodels ", overwrite = TRUE )

## ----loadmodel----------------------------------------------------------------
library("StrathE2E2")
model <- read_model("North_Sea", "1970-1999")

## ----viewmodelobject----------------------------------------------------------
str(model,max.level=2)                    # View the top 2 levels of input list object

## ----runmodel-----------------------------------------------------------------
results <- StrathE2E(model, nyears=5)       # Run the model for 5 years
str(results,max.level=1)                    # View the high-level contents of the output list object

## ----extractendcon, eval=FALSE------------------------------------------------
#  model<-read_model("North_Sea", "2003-2013")              # Read the 2003-2013 internal model
#  results <- StrathE2E(model, nyears=5)                    # Run the model for 5 years
#  extract_new_initial_cond_from_end_of_run(model,results)  # Extract the model endstate as new initial conditions file
#  # This should revert to the results folder since the internal moldel folder is read-only
#  #
#  #
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  copy_model("North_Sea", "2003-2013",
#              dest.path="C:/Users/username/Documents/Models")
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  model<-read_model(model.name="North_Sea",
#                    model.variant="2003-2013",
#                    model.ident="TEST",
#                    user.path="C:/Users/username/Documents/Models")
#  results <- StrathE2E(model, nyears=5)                    # Run the model for 5 years
#  extract_new_initial_cond_from_end_of_run(model,results)  # Extract the model endstate as new initial conditions file
#  # This should write to the user model /Parameters folder)

## ----plot01, fig.width=7, fig.height=6, fig.align='center'--------------------
plot_full_length_timeseries(model, results)

## ----plot01tofile, eval = FALSE-----------------------------------------------
#  pdf("plot.pdf",width=7,height=6)                    # or jpeg("plot.jpg"), png("plot.PNG")
#  plot_full_length_timeseries(model, results)
#  dev.off()

## ----plot02, fig.width=7, fig.height=6, fig.align='center'--------------------
plot_final_year(model, results)

## ----plot03, fig.width=7, fig.height=6, fig.align='center'--------------------
plot_final_year_LTL_inshore_vs_offshore(model, results)

## ----plot04, fig.width=7, fig.height=6, fig.align='center'--------------------
plot_final_year_HTL_inshore_vs_offshore(model, results)

## ----plot05, fig.width=7, fig.height=6, fig.align='center'--------------------
plot_sediment_habitats_final_year(model, results)

## ----plot06, fig.width=7, fig.height=6, fig.align='center'--------------------
box_and_whisker_annual_plots(model, results)

## ----plot07, fig.width=7, fig.height=6, fig.align='center'--------------------
box_and_whisker_monthly_plots(model, results)

## ----plot08, fig.width=8, fig.height=6, fig.align='center'--------------------
plot_time_series_annual_land_disc(model, results)

## ----plot09, fig.width=8, fig.height=6, fig.align='center'--------------------
plot_for_each_gear_catch_by_guild(model, results)

## ----plot010, fig.width=8, fig.height=8, fig.align='center'-------------------
plot_for_each_guild_catch_by_gear(model, results)

## ----viewmodelobject2---------------------------------------------------------
model<-read_model("North_Sea", "2003-2013")
str(model,max.level=2)                    # View the top 2 levels of input list object

## ----viewfleetmodelinput------------------------------------------------------
model<-read_model("North_Sea", "2003-2013")
model$data$fleet.model

## ----runmodelscenario1-1, fig.width=7, fig.height=6, fig.align='center'-------
# Example of code to run a baseline case of the North Sea model with 2003-2013 conditions, and then edit the
# model list object to create a scenario run with the temperature in all zones increased by 2 deg-C
#--------------------------------------------------------
# Read the embedded North Sea 2003-2013 model and assign an identifier for the results
baseTemp_model<-read_model("North_Sea", "2003-2013",model.ident="baseTemp")
# Run the model for 10 years and save the results to a named list object
baseTemp_results<-StrathE2E(baseTemp_model,nyears=10)
# Visualise the output from the run (should show a repeating annual cycle with no trend)
plot_full_length_timeseries(baseTemp_model,baseTemp_results)

## ----runmodelscenario1-2, fig.width=7, fig.height=6, fig.align='center'-------
T_rise<-2    # temperature increase to add to all temperatures in the model drivers
# copy the baseline model list object to a new model list
baseTemp_plusTC_model<-baseTemp_model
# add temperature increase to upper layer offshore temperatures
baseTemp_plusTC_model$data$physics.drivers$so_temp <- baseTemp_model$data$physics.drivers$so_temp+T_rise
# add temperature increase to inshore temperatures
baseTemp_plusTC_model$data$physics.drivers$si_temp <- baseTemp_model$data$physics.drivers$si_temp+T_rise
# add temperature increase to lower layer offshore temperatures
baseTemp_plusTC_model$data$physics.drivers$d_temp  <- baseTemp_model$data$physics.drivers$d_temp+T_rise
# Assign a unique identifier for the .csv outputs
baseTemp_plusTC_model$setup$model.ident <- "baseTemp_plusTC"
# Run the model for 10 years and save the results to a named list object
baseTemp_plusTC_results<-StrathE2E(baseTemp_plusTC_model,nyears=10)
# Visualise the output from the run (should show trends in outputs due to change in T)
plot_full_length_timeseries(baseTemp_plusTC_model,baseTemp_plusTC_results)

## ----runmodelscenario2, eval = FALSE------------------------------------------
#  # Example of code to loop through a set of seven levels of demersal fish harvest ratio ranging from 0 to 3-times
#  # the baseline value for the 2003-2013 North Sea model. The .csv outputs for each level are saved to unique
#  # filenames but the results list-object is discarded. The baseline case is the third level (HScale=1). Each level
#  # is run for 40 years.
#  #--------------------------------------------------------
#  model<-read_model("North_Sea", "2003-2013")  		# Read the embedded North Sea 2003-2013 model
#  for(i in 1:7) {                              		# Start loop through seven model scenarios
#    HScale <- (i-1)*0.5                          		# HScale becomes 0 to 3 in steps of 0.5
#        # Assign a unique identifier for each set of .csv outputs
#    model$setup$model.ident <- paste("Dem_HR_scale_",HScale,sep="")
#        # Set the demersal fish harvest ratio multipler to HScale
#    model$data$fleet.model$HRscale_vector_multiplier[2] <- HScale
#    results<-StrathE2E(model,nyears=40)			 # Run the model, save the .csv results, overwrite any previous list-object
#    print(paste("Harvest Ratio scaling = ",HScale,sep="")) # Print a screen message to monitor progress
#    print("-----------------------------------")		 # Print a screen message to monitor progress
#    plot_full_length_timeseries(model,results)		 # Visualise the output from each run
#  }							 # End loop through scenarios

## ----tornadoplot1, fig.width=6, fig.height=8, fig.align='center'--------------
base_model<-read_model("North_Sea", "1970-1999", model.ident="baseline")  #Read the embedded North Sea 1970-1999 model
base_results <- StrathE2E(base_model,nyears=5)                            # Run the baseline model
# Create a scenario run from the baseline:
scen1_model   <- base_model              # Copy the existing baseline configuration into a new model object
# Edit the model object...
scen1_model$setup$model.ident <- "scenario1"
scen1_model$data$fleet.model$gear_mult[4] <- 0.5        # Gear 4 (Beam_Trawl_BT1+BT2) activity rate rescaled to 0.5*baseline
scen1_results <- StrathE2E(scen1_model,nyears=30)       # Run for 30 years to allow the new model to reach a stationary state
#
# Compare the scenario with the baseline
mdiff_results <- compare_two_runs_aam(model1=NA,results1=base_results,from.csv1=FALSE,
	             		      model2=NA,results2=scen1_results,from.csv2=FALSE,
                     		      log.pc="PC", zone="W",
                     		      bpmin=(-50),bpmax=(+50),
                     	              maintitle="Beam Trawl activity reduced by half")

## ----tornadoplot1-1, fig.width=6, fig.height=8, fig.align='center'------------
# View the list object containing the proportional difference data shown in the plot
# (column LG = log10 values, column PC = percentage values).
# First dataframe in the list = water column components, second = seabed components.  
mdiff_results 
#

## ----tornadoplot2, fig.width=6, fig.height=8, fig.align='center'--------------
# For the previous example, compare the scenario landings and discards with the baseline
cdiff_results <- compare_two_runs_catch(model1=NA,results1=base_results,from.csv1=FALSE,
	               			model2=NA,results2=scen1_results,from.csv2=FALSE,
                       			log.pc="PC", zone="W",
                       			bpmin=(-30),bpmax=(+60),
                       			maintitle="Beam Trawl activity reduced by half")

## ----tornadoplot3, fig.width=6, fig.height=8, fig.align='center'--------------
# View the list object containing the proportional difference data shown in the plot
# (column LG = log10 values, column PC = percentage values).
# First dataframe in the list = landings, second = discards. 
cdiff_results 
#

## ----runmodel3----------------------------------------------------------------
model<-read_model("North_Sea", "2003-2013")
results <- StrathE2E(model, nyears=5)       # Run the model for 5 years
str(results,max.level=1)                    # View the high-level contents of the output list object

## ----resultsobjectdata1-------------------------------------------------------
str(results$build,max.level=1) 

## ----resultsobjectdata2-------------------------------------------------------
head(results$output[,1:6])
tail(results$output[,1:6])

## ----resultsobjectdata3-------------------------------------------------------
# List the names of the first 40 vectors in the results$aggregates list:
names(results$aggregates[1:40])

## ----plotagg, fig.width=6, fig.height=4, fig.align='center'-------------------
# Plot the time series of the sedient nitrate mass in the inshore and offshore zones
# i.e. aggregated across the sediment types in each zone.
# Note that this is the mass not the porewater concentration so uncorrected for
# zonal area or sediment porosity or layer thickness.
plot(seq(1,nrow(results$output)), results$aggregates$x_nitrate_o, type="l",col="black",
     ylim=( c( 0,(1.5*max(results$aggregates$x_nitrate_o)) ) ),
     xlab="Days", ylab="Sediment nitrate mass (mMN)")
lines(seq(1,nrow(results$output)), results$aggregates$x_nitrate_i, type="l",col="red")
legend("topleft",  bg="transparent", c("offshore","inshore"), lty=c(1,1),col=c("black","red"))

## ----resultsobjectdata4-------------------------------------------------------
str(results$fleet.output,max.level=1)

## ----resultsobjectdata5-------------------------------------------------------
str(results$total.annual.catch,max.level=1)

## ----resultsobjectdata6-------------------------------------------------------
str(results$annual.catch.by.gear,max.level=1)

## ----resultsobjectdata7-------------------------------------------------------
str(results$final.year.output,max.level=1)

## ----resultsobjectdata8-------------------------------------------------------
# List the first six entries
str(results$final.year.output[1:6],max.level=1)
#
# View the first two column of the inshore_catchmat array
results$final.year.output$inshore_catchmat[,1:2]

## ----resultsobjectdata9-------------------------------------------------------
results$final.year.output$monthly.averages[,1:2]

## ----resultsobjectdata10------------------------------------------------------
str(results$final.year.output[8:11],max.level=1)
#
head(results$final.year.output$mass_results_inshore)
#
head(results$final.year.output$annual_flux_results_inshore)

## ----resultsobjectdata11------------------------------------------------------
str(results$final.year.output[12:15],max.level=1)

## ----resultsobjectdata12------------------------------------------------------
str(results$final.year.output[16:19],max.level=1)

## ----resultsobjectdata13------------------------------------------------------
str(results$final.year.output[20:22],max.level=1)
#
#Show the first 5 rows and columns of the matrix of all flows:
results$final.year.output$flow_matrix_all_fluxes[1:5,1:5]
#
#Show the first few rows of the dataframe containing outputs from the NetIndices package:
head(results$final.year.output$NetworkIndexResults)

## ----resultsobjectdata14------------------------------------------------------
str(results$final.year.output[23:24],max.level=1)

## ----resultsobjectdata15------------------------------------------------------
str(results$final.year.output[25:27],max.level=1)
#
#Show the overall likelihood value:
results$final.year.output$annual_obj
#
#Show the first few rows of the partial likelihood data:
head(results$final.year.output$partial_chi)
#
#Show the first few rows of the dataframe containing the model outputs corresponding to the annual observational target data:
head(results$final.year.output$opt_results)

## ----processresults1----------------------------------------------------------
out <- results$output

## ----processresults2----------------------------------------------------------
# All the output corresponding to year Y
Y <- 3                    # select year 3
yearY_data <- out[(((Y-1)*360)+1) : ((Y*360)+1),]
#
#List the first few rows and columns of these extracted data
head(yearY_data[,1:6])

## ----processresults3----------------------------------------------------------
# All the output for a given state variable (denoted by "column_name") in year Y:
Y <- 3                    # select year 3
column_name<- "herb_o"    # Select offshore omnivorous zooplankton
col2get <- which(colnames(out)==column_name)
column_data_yearY <- out[(((Y-1)*360)+1) : ((Y*360)+1),col2get]

## ----processresults4----------------------------------------------------------
# Calculate the annual average in year Y of the variable extracted in the previous step:
	mean(column_data_yearY)

## ----processresults5, fig.width=6, fig.height=5, fig.align='center'-----------
column_name<- "phytgrossprod_o"    		   # Select for example offshore gross production by phytoplankton
col2get <- which(colnames(out)==column_name)
J <- nrow(out)					   # number of rows of data in the "out" dataframe
temp <- rep(0,J)				   # creates a vector to hold temporary data
temp[1: (J-1)] <- out[2 : J, col2get]		   # copies data from the "out" into "temp" offset by 1 time increment
temp[J] <- 2*(out[J,col2get]) - out[J-1,col2get]   # fills in the terminal value of the "temp" vector by extrapolation
rate <- temp - out[,col2get]			   # "rate" is a vector of the increments between successive time steps
#
#Plot the original cumulative data and the derived rate
par(mfrow=c(2,1))
par(mar=c(4,4,1,1))
plot(seq(1,length(out[,col2get])),out[,col2get],type="l",xlab="Days",ylab="Cumulative flux")
plot(seq(1,length(rate)),rate,type="l",xlab="Days",ylab="Daily flux")

## ----processresults6----------------------------------------------------------
Y <- 3                                                  # select year 3
column_name<- "phytgrossprod_o"    		        # Select for example offshore gross production by phytoplankton
col2get <- which(colnames(out)==column_name)
out[((Y*360)+1),col2get] - out[(((Y-1)*360)+1),col2get] # integrated flux of the variable denoted by "column_name" over year Y

## ----processresults7----------------------------------------------------------
# Extract the relevant area and volumetric parameters from the $build$model.parameters object
inshore_area 	     <- as.numeric(results$build$model.parameters["shallowprop"])
offshore_upper_thick <- as.numeric(results$build$model.parameters["thik_so"])
offshore_lower_thick <- as.numeric(results$build$model.parameters["thik_d"])
inshore_sed1_area    <- as.numeric(results$build$model.parameters["area_s1"])
inshore_sed1_thick   <- as.numeric(results$build$model.parameters["thik_x_s1"])
inshore_sed1_poros   <- as.numeric(results$build$model.parameters["porosity_s1"])
#
# Time series of suspended detritus and bacteria concentration in the offshore upper layer (mMN.m-3)
detr_bact_conc_so<- out$detritus_so/((1-inshore_area)*offshore_upper_thick)
#
# Time series of the area-density of omnivorous zooplankton in the inshore zone (mMN.m-2)
oz_dens_i<- out$herb_i/inshore_area		
#
# Time series of the depth averaged concentration of omnivorous zooplankton in the offshore zone (mMN.m-3)
oz_conc_o<- out$herb_o /((1-inshore_area)*(offshore_upper_thick+offshore_lower_thick))
#
# Time series of the porewater concentration of ammonia in inshore sediment habitat 1 (mMN.m-3):
xamm_conc_s1<- out$x_ammonia_s1/(inshore_sed1_area*inshore_sed1_thick*inshore_sed1_poros)
#
# Time series of labile detritus & bacterial in inshore sediment habitat 1 as a % of sediment dry weight (%gN.g-1):
xdet_conc_s1<- 100*(out$xR_detritus_s1*(14/1000))/(inshore_sed1_area*inshore_sed1_thick*(1-inshore_sed1_poros)*2650000 )
# (nitrogen atomic weight = 14 g.mole-1; dry sediment density = quartz density = 2.65 x 1e6 g.m-3)	

## ----annealingcode1, eval = FALSE---------------------------------------------
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  	copy_model("North_Sea", "2003-2013",
#  	dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  	model<-read_model(model.name="North_Sea",
#                            model.variant="2003-2013",
#                            model.ident="TEST",
#                            user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick demonstration of the annealing function in operation:
#  annealing_find_ecology_parameters(model, nyears=5, n_iter=10, start_temperature=0.5)
#  #
#  # More realistic configuration would be as follows :
#  annealing_find_ecology_parameters(model, nyears=50, n_iter=1000, start_temperature=1)
#  # (WARNING - this will take about 26 hours to run)
#  #

## ----annealingcode2, eval = FALSE---------------------------------------------
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  	copy_model("North_Sea", "2003-2013",
#  	dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  	model<-read_model(model.name="North_Sea",
#                            model.variant="2003-2013",
#                            model.ident="TEST",
#                            user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick demonstration of the annealing function in operation:
#  annealing_find_harvest_ratio_mult(model, nyears=5, n_iter=10, start_temperature=0.5)
#  #
#  # More realistic configuration would be as follows :
#  annealing_find_harvest_ratio_mult(model, nyears=50, n_iter=1000, start_temperature=1)
#  # (WARNING - this will take about 26 hours to run)
#  #

## ----calculatecode1, eval = TRUE----------------------------------------------
# Load the 2003-2013 version of the North Sea model supplied with the package and calculate scaling parameter values:
model <- read_model("North_Sea", "2003-2013")
#
scale_values <- calculate_hr_scale_values(model)
scale_values
#

## ----annealingcode3, eval = FALSE---------------------------------------------
#  # Copy the 2003-2013 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  copy_model("North_Sea", "2003-2013",
#              dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 2003-2013 version of the North Sea model from the user workspace:
#  model<-read_model(model.name="North_Sea",
#                     model.variant="2003-2013",
#                     model.ident="TEST",
#                     user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick Demonstration of the annealing function in operation:
#  annealing_find_gear_activity_mult_ecosystem_target(model, nyears=5, n_iter=10, start_temperature=0.5)
#  #
#  # More realistic configuration would be:
#  annealing_find_gear_activity_mult_ecosystem_target(model, nyears=50, n_iter=1000, start_temperature=1)
#  #  (WARNING - this will take about 26 hours to run)

## ----annealingcode4, eval = FALSE---------------------------------------------
#  # Copy the 1970-1999 version of the North Sea model supplied with the package into a user workspace (Windows OS):
#  copy_model("North_Sea", "1970-1999",
#              dest.path="C:/Users/username/Documents/Models")
#  #
#  # Load the 1970-1999 version of the North Sea model from the user workspace:
#  model<-read_model(model.name="North_Sea",
#                    model.variant="1970-1999",
#                    model.ident="TEST",
#                    user.path="C:/Users/username/Documents/Models")
#  #
#  # Quick Demonstration of the annealing function in operation:
#  annealing_find_gear_activity_mult_HR_target(model, nchains=5, n_iter=100, start_temperature=0.5)
#  #
#  # More realistic configuration would be :
#  annealing_find_gear_activity_mult_HR_target(model, nchains=100, n_iter=3000, start_temperature=0.5)
#  # (WARNING - this will an hour or so to run)
#  #

## ----sensitivityanalysis1, eval = FALSE---------------------------------------
#  # Load the 2003-2013 version of the North Sea model supplied with the package:
#  model <- read_model("North_Sea", "1970-1999")
#  #
#  # Run the sensitivity analysis process:
#  # WARNING - Running a full sensitivity analysis takes days of computer time on a single machine/processor because it involves a huge number of model runs.
#  # The example below is just a (relatively) quick minimalist demonstration and should NOT be taken as the basis for any analysis or conclusions.
#  # Even so, this minimalist demo could take 45 min to run to completion because it involves 1359 model runs.
#  sens_results <- Sensitivity_analysis_StrathE2E(model, nyears=1, n_iter=3)
#  head(sens_results)
#  #
#  # A more realistic sensitivity analysis would be something like:
#  sens_results <- Sensitivity_analysis_StrathE2E(model, nyears=50, n_iter=16, postprocess=TRUE)
#  # DO NOT launch this configuration unless you are prepared to wait many days for the results
#  #
#  # To share the computational load across two (or more) machines/processors, launch separately on each machine...
#  model1 <- read_model("North_Sea", "1970-1999", model.ident="TEST1")
#  Sensitivity_analysis_StrathE2E(model1, nyears=50, n_iter=10, coldstart=TRUE, postprocess=FALSE)   # On machine 1
#  #
#  model2 <- read_model("North_Sea", "1970-1999", model.ident="TEST2")
#  Sensitivity_analysis_StrathE2E(model2, nyears=50, n_iter=10, coldstart=FALSE, postprocess=FALSE)  # On machine 2
#  # In these cases the function returns a NULL object since postprocess=FALSE but the raw data are still output to CSV
#  #
#  # Then concatenate the two raw results files with text-tags TEST1 and TEST2 afterwards, and post process
#  # the combined file, using the function concatenate_raw_sensitivity_analysis_results()
#  #

## ----sensitivityanalysis2, eval = FALSE---------------------------------------
#  # The example here assumes that three sets of raw results data have been previously
#  # generated by runs of the Sensitivity_analysis_StrathE2E() function, with the model.ident values
#  # assigned as "RUN1", "RUN2" and "RUN3", and that these have been gathered together in the same
#  # /results/Modelname/Variantname folder.
#  # Then... load the model name and version for the data to be combined :
#  # E.g. ... model <- read_model("North_Sea", "2003-2013")
#  #    sens_results <- concatenate_raw_sensitivity_analysis_results(model, ident.list=c("RUN1","RUN2","RUN3"), combined.ident="TESTCOMB", postprocess=TRUE)
#  #    head(sens_results)

## ----sensitivityanalysis3-----------------------------------------------------
# Load the pre-computed data for the 1970-1999 version of the North Sea model supplied with the package:
model <- read_model("North_Sea", "1970-1999")
sens_results <- process_sensitivity_analysis_results_offline(model, use.example=TRUE)
head(sens_results)

## ----sensitivityanalysis4-1, fig.width=7, fig.height=7, fig.align='center'----
# Load the 1970-1999 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Sensitivity analysis and generate the results files
# e.g.   Sensitivity_analysis_StrathE2E(model,nyears=50,n_iter=16)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "1970-1999")
plot_sensitivity_analysis_results(model, use.example=TRUE)

## ----sensitivityanalysis4-2,  eval=FALSE--------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the plot_sensitivity_analysis_results() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=6)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	plot_sensitivity_analysis_results(model)
#  dev.off()

## ----montecarlo1, eval = FALSE------------------------------------------------
#  # Load the 1970-1999 version of the North Sea model supplied with the package and assign an identifier text string:
#  model <- read_model("North_Sea", "1970-1999", model.ident="demo")
#  #Run the Monte Carlo process with demonstration settings and generate the results files
#  Monte_Carlo_StrathE2E(model, nyears=30, n_iter=50)
#  # WARNING - this demonstration run will take about 1 hour.

## ----montecarlopar, fig.width=6, fig.height=10, fig.align='center'------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package:
plot_Monte_Carlo_parameter_distributions(model, use.example=TRUE)

## ----montecarlo2-1,  fig.width=7, fig.height=6, fig.align='center'------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "NUT_PHYT", use.example=TRUE)	# plot nutrient and phytoplankton data

## ----montecarlo2-2,  fig.width=7, fig.height=7, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "SEDIMENT", use.example=TRUE)	# plot sediment chemisty data

## ----montecarlo2-3,  fig.width=6, fig.height=5, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "ZOOPLANKTON", use.example=TRUE)	# plot zooplankton data

## ----montecarlo2-4,  fig.width=7, fig.height=5, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "FISH", use.example=TRUE)		# plot fish and fish larvae data

## ----montecarlo2-5,  fig.width=7, fig.height=5, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "BENTHOS", use.example=TRUE)	# plot benthos and benthos larvae data

## ----montecarlo2-6,  fig.width=7, fig.height=5, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "PREDATORS", use.example=TRUE)	# plot bird, pinniped and cetacean data

## ----montecarlo2-7,  fig.width=6, fig.height=5, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "CORP_DISC", use.example=TRUE)	# plot corpse and discard data

## ----montecarlo2-8,  fig.width=6, fig.height=3, fig.align='center'------------
model <- read_model("North_Sea", "1970-1999")
plot_final_year_time_series_data_with_credible_intervals(model, "MACROPHYTE", use.example=TRUE)	# plot macrophyte data

## ----montecarlo2-9,  eval=FALSE-----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the plot_final_year_data_with_credible_intervals() function call in a graphical device call:
#  # Since the plot pages contain different numbers of panels the recommended width:height ratios are as follows:
#  #	NUT_PHYT 	1.5  : 1,  e.g. width=7,height=4.667
#  #	SEDIMENT 	0.67 : 1,  e.g. width=4,height=6
#  #	ZOOPLANKTON 	1    : 1,  e.g. width=4,height=4
#  #	FISH		2    : 1,  e.g. width=6,height=3
#  #	BENTHOS		2    : 1,  e.g. width=6,height=3
#  #	PREDATORS	2    : 1,  e.g. width=6,height=3
#  #	CORP_DISC	1    : 1,  e.g. width=4,height=4
#  #	MACROPHYTE	2    : 1,  e.g. width=4,height=2
#  # For example:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=7,height=4.667)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	plot_final_year_time_series_data_with_credible_intervals(model, "NUT_PHYT")
#  dev.off()

## ----montecarlo3-1, fig.width=7, fig.height=5, fig.align='center'-------------
# Load the 2003-2013 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "2003-2013")
plot_final_year_migration_data_with_credible_intervals(model, use.example=TRUE)

## ----montecarlo3-2,  eval=FALSE-----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the plot_final_year_migration_data_with_credible_intervals() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=4)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	plot_final_year_migration_data_with_credible_intervals(model)
#  dev.off()

## ----montecarlo4-1, fig.width=6, fig.height=7, fig.align='center'-------------
# Load the 2003-2013 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "2003-2013")
plot_final_year_trophic_data_with_credible_intervals(model, use.example=TRUE)

## ----montecarlo4-2,  eval=FALSE-----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the final_year_trophic_data_with_credible_intervals() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=4,height=6)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	plot_final_year_trophic_data_with_credible_intervals(model)
#  dev.off()

## ----montecarlo5-1, fig.width=6, fig.height=5, fig.align='center'-------------
# Load the 2003-2013 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "2003-2013")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "2003-2013")
plot_inshore_vs_offshore_anavmass_with_credible_intervals(model, use.example=TRUE)

## ----montecarlo5-2, eval = FALSE----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the plot_inshore_vs_offshore_anavmass_with_credible_intervals() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=4)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	plot_inshore_vs_offshore_anavmass_with_credible_intervals(model)
#  dev.off()

## ----montecarlo6-1, fig.width=7, fig.height=9, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "1970-1999")
box_and_whisker_annual_plots_with_credible_intervals(model, use.example=TRUE)

## ----montecarlo6-2, eval = FALSE----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the box_and_whisker_annual_plots_with_credible_intervals() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=6)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	box_and_whisker_annual_plots_with_credible_intervals(model)
#  dev.off()

## ----montecarlo7-1, fig.width=7, fig.height=6, fig.align='center'-------------
# Load the 1970-1999 version of the North Sea model supplied with the package:
# model <- read_model("North_Sea", "1970-1999")
# Run the Monte Carlo process and generate the results files
# e.g.   Monte_Carlo_StrathE2E(model,nyears=50,n_iter=500)
#
# ..or
#
# Read the example data provided with the package :
model <- read_model("North_Sea", "1970-1999")
box_and_whisker_monthly_plots_with_credible_intervals(model, use.example=TRUE)

## ----montecarlo7-2, eval = FALSE----------------------------------------------
#  # To direct the graph output to a file rather than the screen, wrap the box_and_whisker_monthly_plots_with_credible_intervals() function call in a graphical device call:
#  pdf("C:/Users/username/Documents/Foldername/plot.pdf",width=6,height=5)      # or jpeg("plot.jpg"), png("plot.PNG")
#  	box_and_whisker_monthly_plots_with_credible_intervals(model)
#  dev.off()

