#
# e2e_plot_trophic.R
#
#' Plot showing the annual mean trophic level index, and the omnivory index of each guild during the final year of a model run, optionally with credible intervals.
#'
#' Generate a two-panel plot showing: (upper panel) the mean trophic level of each guild in the ecology model, and (lower panel) the omnivory index of each guild. The data are 
#' generated by the NetIndices package from a flow matrix of nutrient fluxes through, into and out of the ecosystem during the final year of a run. The data are generated
#' automatically as part of the output from every call of the e2e_run() function. The default is to plot data from a single model run but if available, 
#' credible intervals of model output from a Monte Carlo analysis can be plotted instead.
#'
#' If credible intervals are plotted these are displayed as box-and-whiskers. The box spans 50% of the likelihood distribution of values and the whiskers 99%. The median is shown 
#' by a black tick mark and the maximum likelihhod model by a red tick mark.
#'
#' Arguments determine the source of model data to be plotted. These can be outputs from a single model run with data held in memory as a list object or in a saved csv file, or from 
#' a Monte Carlo simulation (using the function e2e_run_mc()) to estimate credible intervals of model outputs. Generation of credible interval data is a long computing task, so 
#' example data for the North Sea model provided with the package are available as illustration.
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2e_read() function.
#' @param ci.data Logical. If TRUE plot credible intervals around model results based on Monte Carlo simulation with the e2e_run_mc() function (default=FALSE).
#' @param use.saved Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).
#' @param use.example Logical. If TRUE use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE).
#' @param results R-list object of model output generated by the e2e_run() function. Only needed if ci.data=FALSE, use.saved=FALSE and use.example=FALSE. (Default=NULL).
#'
#' @return Graphical display in a new graphics window.
#'
#' @seealso \code{\link{e2e_read}}, \code{\link{e2e_run}}, \code{\link{e2e_run_mc}}, \code{\link{e2e_plot_migration}}, \code{\link{e2e_plot_catch}}, \code{\link{e2e_plot_eco}}, \code{\link{e2e_plot_biomass}}
#'
#' @importFrom graphics text
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the North Sea model supplied with the package, run, and
#' # generate a plot:
#' model <- e2e_read("North_Sea", "1970-1999")
#' results <- e2e_run(model, nyears=3)
#' e2e_plot_trophic(model, results=results)
#' # Alternatively, plot the same data from a csv file saved by the e2e_run() function:
#' dev.new()
#' e2e_plot_trophic(model, use.saved=TRUE)
#'
#' # For the same model as supplied with the package, plot the example data with credible intervals:
#' dev.new()
#' e2e_plot_trophic(model, ci.data=TRUE, use.example=TRUE)
#'
#' # Direct the graphics output to a file (Windows OS)... 
#' # or jpeg("plot.jpg"), png("plot.png")
#' pdf("plot.pdf",width=6,height=8)
#' e2e_plot_trophic(model, ci.data=TRUE, use.example=TRUE)
#' dev.off()
#'
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

e2e_plot_trophic <- function(model,ci.data=FALSE, use.saved=FALSE, use.example=FALSE, results=NULL  ) {

	if (use.example == TRUE) hasExamples()

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

if(use.saved==TRUE & use.example==TRUE){
        stop("use.saved and use.example cannot both be TRUE ! \n")
}

if(use.saved==TRUE & is.list(results)==TRUE){
        stop("use.saved is TRUE but a dataframe object has also been specified ! \n")
}

if(use.example==TRUE & is.list(results)==TRUE){
        stop("use.example is TRUE but a dataframe object has also been specified ! \n")
}

if(use.saved==FALSE & use.example==FALSE & is.list(results)==FALSE){
        stop("no source of data has been specified ! \n")
}

if(ci.data==FALSE & use.example==TRUE){
        stop("no example data available for a single model run - run the model to generate some data ! \n")
}

if(ci.data==TRUE & use.saved==FALSE & use.example==FALSE & is.list(results)==TRUE){
        stop("credible intervals available only from saved or example data ! \n")
}

#.........................................................................

if(ci.data == FALSE & use.saved==FALSE & is.list(results)==TRUE ){
	plot_final_year_trophic_data(model, use.saved=FALSE,results=results)
}

if(ci.data == FALSE & use.saved==TRUE ){
	plot_final_year_trophic_data(model, use.saved=TRUE)
}

if(ci.data == TRUE & use.saved==FALSE & use.example==TRUE ){
	plot_final_year_trophic_data_with_ci(model, use.example=TRUE)
}

if(ci.data == TRUE & use.saved==TRUE & use.example==FALSE ){
	plot_final_year_trophic_data_with_ci(model, use.example=FALSE)
}

#.........................................................................


}

#-----------------------------------------------------------------


