#
# postprocess_sens_data.R
#
#' Read and post-process raw data saved as a csv file from the e2e_run_sens() function 
#'
#' Reads raw data generated by the function e2e_run_sens() from a saved CSV file, and creates a post-processed output file
#' containing the mean Elementary Effect (EE_mean) and the standard deviation of EE (EE_sd) for each parameter, sorted by the absolute value of EE_mean.
#' EE_mean is an index of the magnitude of the sensitivity of a parameter, and EE_sd is an index of the extent of interaction with other parameters.
#'
#' The raw data input is a table of Elementary Effect values for each run of the model. Model runs are organised by trajectories which consist
#' of a baseline run in which all parameters are perturbed from an initial state by the addition of terms drawn from a random normal of mean zero, plus
#' a set of runs in which each parameter in turn is perturbed by a fixed amount.
#'
#' The function reads the file of raw sensitivity analysis data (OAT_results-*.csv, where * refers to the identifier model.ident set in the e2e_read() function) from a /results folder
#' and performs the post-processing that would ordinarily be done automatically within the e2e_run_sens() or 
#' e2e_merge_outputs() functions. The function is provided as a backup to mitigate against data loss in the event of some interruption to the normal
#' automatic data processing since running a sensitivity analysis represents a significant investment of computer time.
#'
#' Optionally, the function can read example data for one of the two North Sea model variants supplied with the package.
#'
#' Output from the function is a processed data file named sorted_parameter_elementary_effects-*.csv in the current folder /results/Modelname/Variantname/
#'
#' For details of how the Elementary Effect values are derived for each parameter see help(e2e_run_sens).
#'
#' @references Morris, M.D. (1991). Factorial sampling plans for preliminary computational experiments. Technometrics, 33, 161-174.
#'
#' @param model R-list object defining the model configuration compiled by the e2e_read() function
#' @param use.example (TRUE or FALSE) Option to use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE)
#' @param csv.output Logical. If FALSE the disable writing of CSV output files - useful for testing (default=TRUE)
#'
#' @return Datraframe and csv file of processed output from sensitivity analysis
#'
#' @noRd
#
# ------------------------------------------------------------------------------

postprocess_sens_data <- function(model, use.example=FALSE,csv.output=TRUE) {

	pkg.env$csv.output <- csv.output	# controls writing of CSV files

	resultsdir <- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")
	model.name <- elt(model, "setup", "model.name")
	model.variant <- elt(model, "setup", "model.variant")

if(use.example==TRUE){
	#sensfile <- examplecsvname(model.name, model.variant, "OAT_results")
	#check.exists(sensfile)
	#print(paste("Reading internally stored example results for the ",model.name," ",model.variant," model",sep=""))
	print(paste("Output wil be directed to the folder ",resultsdir))
	#results_df_out<- readcsv(sensfile)
	results_df_out <- get.example.results(model.name, model.variant, "OAT_results")
}

if(use.example==FALSE){
	sensfile<- csvname(resultsdir, "OAT_results", model.ident)
	check.exists(sensfile)
	results_df_out<- readcsv(sensfile)
}

	Sorted_SENS_results <- process_sensitivity_analysis_results(model, results_df_out)

	return(Sorted_SENS_results)

}

#----------------------------------------------------------------------
