#
# e2e_extract_start.R
#
#' Extract the values of all the state variables at the end of a model run and format for use as new initial conditions.
#'
#' The function saves the state of the model at the end of a run (using the e2e_run() function) for use as initial conditions in future runs. This enables, 
#' for example, the model to be run for a long time to attain a stationary state, and then restarted in that state.
#'
#' Initial conditions for a model run are held in the /Param folder of the Model/Variant path specified in the e2e_read()
#' function call used to define a run. By default, the function attampts to write the model end-state file back to this /Param folder.
#' However, the package folders are read-only so if e2e_read() has been specified to load an internally provided Model/Variant then 
#' the output will revert to the currently specified results folder instead. To fix this, copy the required package model to a user
#' workspace using the e2e_copy() function and re-run.
#'
#' The new initial conditions file will have a name initial_values-*.csv, where * is the model.ident text identifier specified in e2e_read()
#' To source the new initial conditions in a subsequent model run, edit the MODEL_SETUP.csv file in the required /Models/Variant folder
#'
#' @param model R-list object defining the model configuration compiled by the e2e_read() function.
#' @param results R-list object generated by the e2e_run() function.
#' @param csv.output Logical. If FALSE then disable writing of csv output files - useful for testing (default=TRUE).
#'
#' @return Table of state variable values from the end of a run formatted as required for input to a new model run as initial conditions. By default these data are also returned as a csv file to the Param folder of the current model, unless this is one of the North Sea model versions within the package.
#'
#' @seealso \code{\link{e2e_read}}, \code{\link{e2e_run}} 
#'
#' @export
#'
#' @examples
#' # Copy the 2003-2013 version of the North Sea model supplied with the package to a
#' # user workspace (Windows OS):
#' \dontrun{
#' ee2_copy("North_Sea", "2003-2013",
#'            dest.path="C:/Users/username/Documents/Models")
#' model <- e2e_read("North_Sea", "2003-2013", 
#'	      user.path="C:/Users/username/Documents/Models", model.ident="TEST")
#' results <- e2e_run(model, nyears=5)
#' new_initial <- e2e_extract_start(model,results)
#' }
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------


e2e_extract_start <- function(model, results,csv.output=TRUE) {

	pkg.env$csv.output <- csv.output	# controls writing of CSV files


	setup		<- elt(model, "setup")
	read.only	<- elt(setup, "read.only")
	model.path	<- elt(setup, "model.path")
	resultsdir	<- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")

	output<-results$output

	endstate <- data.frame(rep(0,77))
	for(jj in 2:78) {
		endstate[jj-1, 1] <- output[nrow(output), jj]
	}
	rownames(endstate)<-names(output[,2:78])

	if (read.only & csv.output==TRUE) {
		cat("Warning: cannot write model end-state back to the model input folders - model is read-only\n")
		cat("Warning: to fix this, make a copy of the model using e2e_copy() into your own workspace.\n")
		cat("Warning: writing the model end-state file to the current results folder instead.\n")
		#stop("Model is not writable!")
	}

	if (read.only) {
		filename = csvname(resultsdir, "initial_values", model.ident)
	} else {
		parameterpath <- makepath(model.path, PARAMETERS_DIR)
		filename = csvname(parameterpath,  "initial_values", model.ident)
	}

	writecsv(endstate, filename, header=FALSE)

        names(endstate)<-""
	return(endstate)

}

#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
