#
# e2e_plot_catch.R
#
#' Plot the distribution and composition of annual catches across guild, zones and gears in the final year of a model run.
#'
#' Create stacked barplots of the distribution of offshore and inshore landings and discards across gears by guild, or across guilds by gears, in the final year of a model run.
#'
#' Data in zonal landings and discards by guild and gear in the final year of a model run are generated as a standard output from the model, and saved both as csv files and in the results object returned by the e2e_run() function.
#' This function organises these data in two different ways to display as barplots.
#'
#' The first display is a multi-panel plot in which each panel represents a different guild, and the bars show the zonal landings by each gear. The alternative display
#' has each panel as a different gear, and the bars show the landings and discards of each guild.
#'
#' The unit of the displayed data are mMN/y from the model domain as a whole, which is taken as being 1m2.
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2e_read() function.
#' @param results List object of single-run model output generated by running the function e2e_run() function.
#' @param selection Text string from a list identifying the group of model output variables to be plotted. Select from: "BY_GUILD", "BY_GEAR". With the former, each panel represents a different guild; with the latter, each panel represnts a different gear. Remember to include the phrase within "" quotes.
#'
#' @return graphical display in a new graphics window.
#'
#' @seealso \code{\link{e2e_read}}, \code{\link{e2e_run}}, \code{\link{e2e_run_mc}}, \code{\link{e2e_plot_migration}}, \code{\link{e2e_plot_eco}}, \code{\link{e2e_plot_trophic}}, \code{\link{e2e_plot_biomass}}
#'
#' @importFrom graphics text
#'
#' @export
#'
#' @examples
#' # Load the 1970-1999 version of the North Sea model supplied with the package, run, and
#' # generate a plot:
#' model <- e2e_read("North_Sea", "1970-1999")
#' results <- e2e_run(model, nyears=3)
#' e2e_plot_catch(model, results, selection="BY_GEAR")
#' dev.new()
#' e2e_plot_catch(model, results, selection="BY_GUILD")
#'
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------


e2e_plot_catch <- function(model, results, selection="BY_GEAR") {

if(selection=="BY_GUILD"){
	plot_for_each_guild_catch_by_gear(model=model,results=results)

} else if(selection=="BY_GEAR"){
	plot_for_each_gear_catch_by_guild(model=model,results=results)

} else {
	stop("Error: unknown selection '", selection, "' !\n")
}

}

#-----------------------------------------------------------------


