% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2e_optimize_eco.R
\name{e2e_optimize_eco}
\alias{e2e_optimize_eco}
\title{Optimize StrathE2E ecology model parameters to maximise the likelihood of observed ecosystem target data.}
\usage{
e2e_optimize_eco(model, nyears = 40, n_iter = 500,
  start_temperature = 1, cooling = 0.975, toppredlock = TRUE,
  quiet = TRUE, csv.output = TRUE, runtime.plot = TRUE)
}
\arguments{
\item{model}{R-list object generated by the e2e_read() function which defined the model configuration.}

\item{nyears}{Number of years to run the model in each iteration (default=40).}

\item{n_iter}{Number of iterations of the model (default=500).}

\item{start_temperature}{Initial value of the simulated annealing temperature parameter (default=1). Suggested values in the range 0.0005 - 5. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood.}

\item{cooling}{Rate at which the simulated annealing temperature declines with iterations (default=0.975). Suggested values in the range 0.9 - 0.985}

\item{toppredlock}{Logical. If TRUE then locks-down the uptake parameters of the birds pinnipeds and cetaceans as these are hard to fit alongside the other parameters (default=TRUE).}

\item{quiet}{Logical. If TRUE then suppress informational messages at the start of each iteration (default=TRUE).}

\item{csv.output}{Logical. If FALSE then disable writing of csv output files - useful for testing (default=TRUE).}

\item{runtime.plot}{Logical. If FALSE then disable runtime plotting of the progress of the run - useful for testing (default=TRUE)}
}
\value{
A list object containing the histories of proposed and accepted parameters and the final accepted parameter values. Optionally (by default), csv files of the histories and the final accepted parameter values. The latter are returned to the model parameter folder in a format to be read back into the model.
}
\description{
Launches a StrathE2E simulated annealing process to find the set of ecology model parameters
producing the maximum likelihood of observed target data on the state of the ecosystem,
given specified environmental driving data and fishing fleet parameters.
}
\details{
Simulated annealing is is a probabilistic technique for approximating the global optimum of a given function. As implemented here the process searches the parameter space
of a model to locate the combination which maximises the likelihood of a set of observed
data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1).

Model configuration and initial values of the ecology model parameters need to be
assembled by a prior call of the e2e_read() function.

NOTE that the user.path argument in the e2e_read() function call needs to point to a user workspace folder, not the default
North Sea model provided with the package. This is because the annealing function needs write-access to the model /Param folder,
but the /extdata/Models folder in the package installation is read-only.
To use the annealing function on the North Sea model, use the e2e_copy() function to make a copy of the
North Sea model in the user workspace.

The observational data to which the ecology parameters are optimized are loaded from the folder Modelname/Variantname/Target/annual_observed_*.csv as part of a e2e_read() function call and are built into the R-list object generated by e2e_read().
Column 3 of annual_observed_* (header: "Use1_0") is a flag to set whether any given row is used in calculating the likelihood of the observed data given the model setup and parameters. Un-used rows of data are omitted from calculations.

The coefficients of variation for jiggling the ecology parameter can be varied in real-time
during the run by editing the file "optimize_ecology.csv" in the folder
/Param/control/ of the model version.

The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
the likelihoods of the proposed and accepted parameter sets at each iteration.
y-axis (likelihood of the target data) range of the real time plot can be varied during the run by 
editing the setup file "optimize_ecology.csv"

At the end of the procedure new versions of the three ecology model 'fitted_parameters..' files are exported
to the folder /Param of the model version, with a user defined identifier specified by the model.ident argument
in the e2e_read() function.

In order to use the new fitted parameter values in a subsequent run of the StrathE2E model (using the e2e_run() function)
it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new files.

Also at the end of the procedure the proposed and accepted ecology model parameter values and corresponding likleihoods from each iteration of the procedure are saved as CSV files 
in the results folder and in a list object which is returned by the function. 
The two csv files generated by the procedure have names:  
annealing_par_proposalhistory-*, annealing_par_acceptedhistory-*, where * denotes the value
of model.ident defined in the preceding e2e_read() function call.
The returned list object contains three dataframes: parameter_proposal_history, parameter_accepted_history, new_parameter_data (a list of three).
The proposal and accepted histories can be further analysed with the function e2e_plot_opt_diagnostics() to assess the performance of the optimization process.
}
\examples{
\dontrun{
# Load the 1970-1999 version of the North Sea model supplied with the package and generate a
# quick test data object with only 10 itereations and running the model for 5 years.
# More realistic would be at least 500 iterations and running for 50 years.
# Even so this example will take a few minutes to run:
model<-e2e_read(model.name="North_Sea",
                model.variant="1970-1999",
                model.ident="test")
# This model is already optimized to the observed ecosystem data supplied with the package
# Perturb the temperature driving to knock the model away from its maximum likelihood
# state relative to the target data:
# add 3 degC to upper layer offshore temperatures:
model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3
# add 3 degC to inshore temperatures:
model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3
# add 3 degC to lower layer offshore temperatures:
model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3
test_run  <- e2e_optimize_eco(model, nyears=5, n_iter=10, start_temperature=0.4,csv.output=FALSE)
# View the structure of the returned list:
str(test_run,max.level=1)
# View the structure of the returned list element containing parameter objects:
str(test_run$new_parameter_data,max.level=1)
# View the new preference matrix:
test_run$new_parameter_data$new_preference_matrix

# More realistic configuration: (WARNING - this will take about 26 hours to run) :
# Copy the 1970-1999 version of the North Sea model supplied with the package into a
# user workspace (Windows OS):
e2e_copy("North_Sea", "1970-1999",
           dest.path="C:/Users/username/Documents/Models")
# Load the 1970-1999 version of the North Sea model from the user workspace:
model<-e2e_read(model.name="North_Sea",
                model.variant="1970-1999",
                model.ident="fittingrun",
                user.path="C:/Users/username/Documents/Models")
# Launch the fitting process
fitting_data <- e2e_optimize_eco(model, nyears=50, n_iter=1000, start_temperature=1)
}

}
\seealso{
\code{\link{e2e_ls}} , \code{\link{e2e_read}} , \code{\link{e2e_plot_opt_diagnostics}} , \code{\link{e2e_optimize_hr}} , \code{\link{e2e_optimize_act}}
}
