% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2e_read.R
\name{e2e_read}
\alias{e2e_read}
\title{Read all the configuration, driving and parameter files for a designated model and compile the data into a list object.}
\usage{
e2e_read(model.name, model.variant, model.ident = "base",
  model.subdir = "", user.path = "", quiet = TRUE)
}
\arguments{
\item{model.name}{Name of model to read (no default).}

\item{model.variant}{Variant of the model to be read (no default).}

\item{model.ident}{Text appended to output file names (e.g. OFFSHORE_model_annualresults-TEXT.csv instead of just OFFSHORE_model_annualresults.csv) (default="base").}

\item{model.subdir}{Sub directory of the main results folder for storing results (default = a folder which will be auto-created using the model name).}

\item{user.path}{Path to users top level model folder from where the model setup will be loaded (default = load from the package folder extdata/Models).}

\item{quiet}{Logical. If FALSE then see on-screen information on where model is loaded from and what files are read (default=TRUE).}
}
\value{
R-list object containing all of the model configuration data.
}
\description{
This function is the key point of entry to the package. It reads all the csv files containing the configuration, driving and parameter values which define
a model for a particular region and compiles them into a single R-list object. This data object forms the basis for almost all other functions in the package.
}
\details{
The input csv files are specified in the MODEL_SETUP.csv file located
in the Model/variant folder specified by the argument user.path in a e2e_read() call, or from the extdata/Models folder
in the package installation. The models supplied with the package are two variants (1970-1999 and 2003-2013) of a fully
parameterised and documented model of the North Sea.

Starting from a baseline model configuration defined in the MODEL_SETUP.csv file, any of the terms in the R-list objcet created by e2e_read() can be modified by coding
to produce an unlimited range of scenario configurations representing, for example, changes in the physical environment, changes in the composition or activity of the
fishing fleets, or any combination of these. The power of this approach is that large numbers of scenarios can be coded in standard R and simulated without any need
for manual editing of input files.
}
\examples{
# Load the 2003-2013 version of the North Sea model supplied with the package and set a value
# for the identifier to be added to output file names; outputs go to a default results folder;
# on-screen information enabled:
model <- e2e_read("North_Sea", "2003-2013",model.ident="demo",quiet=FALSE)

\dontrun{
# Load a user defined model from a user workspace (Windows OS):
model<-e2e_read("Modelname", "Variantname",
                 model.ident="demo",
                 user.path="C:/Users/username/Documents/Foldername/Models")

# Load a user defined model from a user workspace and send the model outputs to
# a specified folder (Windows OS):
model<-e2e_read("Modelname", "Variantname",
                 model.ident="demo",
                 model.subdir="C:/Users/username/Documents/Foldername/results",
                 user.path="C:/Users/username/Documents/Foldername/Models")

# View details of the structure and contents of the model list-object
str(model,max.level=1)
str(model,max.level=2)

# Create a new scenario version of the North Sea/1970-1999 model by increasing the
# activity rate of gear 1 (pelagic trawls and seines) by a factor of 2
model <- e2e_read("North_Sea", "1970-1999",model.ident="Gear1x2")
# Change the gear activity multiplier for gear 1 to a value of 2:
model$data$fleet.model$gear_mult[1] <- 2
# Set a new identifier for the outputs:
model$setup$model.ident <- "gear1x2"

# Load a baseline case of the North Sea model with 2003-2013 conditions, and then create a
# scenario with the temperature in all zones increased by 1C
# Read the embedded North Sea 2003-2013 model and assign an identifier for the results
baseTemp_model<-e2e_read("North_Sea", "2003-2013",model.ident="baseTemp")
# Run the model and save the results to a named list object:
baseTemp_results<-e2e_run(baseTemp_model,nyears=40)
# Visualise the tull-length time series of output from the run:
e2e_plot_ts(baseTemp_model,baseTemp_results,selection="ECO")
# copy the baseline model list object to a new model list:
baseTemp_plus1C_model<-baseTemp_model
# add 1 degC to upper layer offshore temperatures:
baseTemp_plus1C_model$data$physics.drivers$so_temp<- baseTemp_model$data$physics.drivers$so_temp+1
# add 1 degC to inshore temperatures:
baseTemp_plus1C_model$data$physics.drivers$si_temp<- baseTemp_model$data$physics.drivers$si_temp+1
# add 1 degC to lower layer offshore temperatures:
baseTemp_plus1C_model$data$physics.drivers$d_temp <- baseTemp_model$data$physics.drivers$d_temp+1
# Assign a unique identifier to each set of .csv outputs:
baseTemp_plus1C_model$setup$model.ident <- "baseTemp_plus1C"
# Run the model and save the results to a named list object:
baseTemp_plus1C_results<-e2e_run(baseTemp_plus1C_model,nyears=40)
dev.new()
# Open a new graphics window
e2e_plot_ts(baseTemp_plus1C_model,baseTemp_plus1C_results,selection="ECO")
# Visualise the full-length time series of output from the run
}
}
\seealso{
\code{\link{e2e_ls}}, \code{\link{e2e_copy}}, \code{\link{e2e_run}}
}
