#
# e2e_optimize_hr.R
#
#' Optimize StrathE2E harvest ratio multipliers to maximum the likelihood of observed ecosystem target data.
#'
#' Launches a StrathE2E simulated annealing process to find the set of fishing fleet model harvest ratio multipliers
#' producing the maximum likelihood of observed target data on the state of the ecosystem,
#' given specified environmental driving data and ecology model parameters.
#'
#' Simulated annealing is is a probabilistic technique for approximating the global optimum of a given function. As implemented here the process searches the parameter space
#' of a model to locate the combination which maximises the likelihood of a set of observed
#' data corresponding to a suite of derived outputs. Parameter combinations which result in an improved likelihood
#' may be rejected according to a probability ('temperature') which decreases as the iterations progress. This is to  
#' avoid becoming stuck at local likelihood-maxima. The rate at which the 'temperature' decreases is set
#' by a 'cooling' parameter (fraction of previous temperature at each iteration, 0<value<1).
#'
#' Model configuration and initial values of the ecology model parameters need to be
#' assembled by a prior call of the e2e_read() function.
#' 
#' NOTE that the user.path argument in the e2e_read() function call needs to point to a user workspace folder, not the default
#' North Sea model provided with the package. This is because the annealing function needs write-access to the model /Param folder,
#' but the /extdata/Models folder in the package installation is read-only.
#' To use the annealing function on the North Sea model, use the e2e_copy() function to make a copy of the
#' North Sea model in the user workspace.
#'
#' The coefficient of variation for jiggling the harvest ratio multipliers can be varied in real-time
#' during the run by editing the file "optimize_fishing.csv" in the folder
#' /Param/control/ of the model version. Suggested vaues for the CV are in the range 0.1 to 0.01
#'
#' The observational data to which the harvest ratio multiplier parameters are optimized are loaded from the folder Modelname/Variantname/Target/annual_observed_*.csv as part of a e2e_read() function call and are built into the R-list object generated by e2e_read().
#' Column 3 of annual_observed_* (header: "Use1_0") is a flag to set whether any given row is used in calculating the likelihood of the observed data given the model setup and parameters. Un-used rows of data are omitted from calculations.
#'
#' The function produces a real-time graphical summary of the progress of the fitting procedure, displaying
#' the likelihoods of the proposed and accepted parameter sets at each iteration.
#' y-axis (likelihood of the target data) range of the real time plot can be varied during the run by 
#' editing the setup file "optimize_fishing.csv"
#'
#' At the end of the procedure a new version of the harvest ratio multipliers file is exported
#' to the folder /Param of the model version, with a user defined identifier specified by the model.ident argument
#' in the e2e_read() function. The histories of proposed and accepted parameter combinations
#' are saved as csv files in the results folder.
#'
#' To preserve the new harvest ratio multipliers and incorporate them into the fishing fleet model parameterisation
#' the multiplier values need to be applied to the scaling parameters which link the integrated effort by each gear to the harvest ratio
#' value which gets piped into the ecology model. Manually update the values in rows 12-21 (excluding the header row) of the
#' file /Param/fishing_fleet_*.csv, by multiplying the existing values by the new multipliers emerging from the annealing process.
#'
#' If the edited file 'fishing_fleet_*.csv' is saved with a new identifier (*) then in order to use it in a subsequent run of the
#' StrathE2E model (using the e2e_run() function) it will be necessary to edit the MODEL_SETUP.csv file in the relevant /Models/variant folder to point to the new file.
#'
#' Also at the end of the procedure the proposed and accepted harvest ratio multipier values and corresponding likleihoods from each iteration of the procedure are saved as CSV files 
#' in the results folder and in a list object which is returned by the function. 
#' The two csv files generated by the procedure have names:  
#' annealing_HRmult_proposalhistory-*, annealing_HRmult_acceptedhistory-*, where * denotes the value
#' of model.ident defined in the preceding e2e_read() function call.
#' The returned list object contains three dataframes: parameter_proposal_history, parameter_accepted_history, new_parameter_data (a list of three).
#' The proposal and accepted histories can be further analysed with the function e2e_plot_opt_diagnostics() to assess the performance of the optimization process.
#'
#' @param model R-list object generated by the e2e_read() function which defined the model configuration.
#' @param nyears Number of years to run the model in each iteration (default=40).
#' @param n_iter Number of iterations of the model (default=500).
#' @param start_temperature Initial value of the simulated annealing temperature parameter (default=1). Suggested values in the range 0.0005 - 5. Higher values increase the probability of rejecting parameter combinations producing an improvement in likelihood.
#' @param cooling Rate at which the simulated annealing temperature declines with iterations (default=0.975). Suggested values in the range 0.9 - 0.985
#' @param quiet Logical. If TRUE then suppress informational messages at the start of each iteration (default=TRUE).
#' @param csv.output Logical. If FALSE then disable writing of csv output files - useful for testing (default=TRUE).
#' @param runtime.plot Logical. If FALSE then disable runtime plotting of the progress of the run - useful for testing (default=TRUE).
#'
#' @return A list object containing the histories of proposed and accepted parameter values and the final accepted parameter values. Optionally (by default), csv files of the histories and the final accepted parameter values. The latter are returned to the model parameter folder in a format to be read back into the model.
#'
#' @seealso \code{\link{e2e_ls}} , \code{\link{e2e_read}} , \code{\link{e2e_plot_opt_diagnostics}} , \code{\link{e2e_optimize_eco}} , \code{\link{e2e_optimize_act}} 
#'
#' @importFrom stats runif
#' @importFrom grDevices dev.off
#'
#' @export
#'
#' @examples
#' \dontrun{
#' model<-e2e_read(model.name="North_Sea",
#'                 model.variant="1970-1999",
#'                 model.ident="test")
#' # This model is already optimized to the observed ecosystem data supplied with the package
#' # Perturb the temperature driving to knock the model away from its maximum likelihood
#' # state relative to the target data:
#' # add 3 degC to upper layer offshore temperatures:
#' model$data$physics.drivers$so_temp <- model$data$physics.drivers$so_temp+3
#' # add 3 degC to inshore temperatures:
#' model$data$physics.drivers$si_temp <- model$data$physics.drivers$si_temp+3
#' # add 3 degC to lower layer offshore temperatures:
#' model$data$physics.drivers$d_temp  <- model$data$physics.drivers$d_temp+3
#' test_run  <- e2e_optimize_hr(model, nyears=5, n_iter=10, start_temperature=0.4,csv.output=FALSE)
#' # View the structure of the returned list:
#' str(test_run,max.level=1)
#' # View the structure of the returned list element containing parameter objects:
#' str(test_run$new_parameter_data,max.level=1)
#' # View the new, final accepted parameter data:
#' test_run$new_parameter_data
#'
#' # More realistic configuration would be (WARNING - this will take about 26 hours to run) :
#' # Copy the 1970-1999 version of the North Sea model supplied with the package into a
#' # user workspace (Windows OS):
#' e2e_copy("North_Sea", "1970-1999",
#'           dest.path="C:/Users/username/Documents/Models")
#' # Load the 1970-1999 version of the North Sea model from the user workspace:
#' model<-e2e_read(model.name="North_Sea",
#'                 model.variant="1970-1999",
#'                 model.ident="fittingrun",
#'                 user.path="C:/Users/username/Documents/Models")
#' # Launch the fitting process
#' e2e_optimize_hr(model, nyears=50, n_iter=1000, start_temperature=1)
#' }
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------


e2e_optimize_hr <- function(model, nyears=40, n_iter=500, start_temperature=1, cooling=0.975, quiet=TRUE, csv.output=TRUE, runtime.plot=TRUE) {

	if(runtime.plot==TRUE){
	start_par = par()$mfrow
	dev.off()
	on.exit(par(mfrow = start_par))
	}

	pkg.env$csv.output <- csv.output	# controls writing of CSV files

	setup		<- elt(model, "setup")
	read.only	<- elt(setup, "read.only")
	model.path	<- elt(setup, "model.path")
	resultsdir	<- elt(setup, "resultsdir")
	identifier	<- elt(setup, "model.ident")

	data				<- elt(model, "data")
	fleet.model			<- elt(data, "fleet.model")
	fitted.parameters		<- elt(data, "fitted.parameters")
	gear_mult			<- elt(fleet.model, "gear_mult")
	HRscale_vector_multiplier	<- elt(fleet.model, "HRscale_vector_multiplier")

	if (read.only & csv.output==TRUE) {
		cat("Warning: cannot write fitted parameters back to the model input folders - model is read-only\n")
		cat("Warning: to fix this, make a copy of the model using e2e_copy() into your own workspace.\n")
		stop("Model is not writable!")
	}

	print(date())

	if (length(which(HRscale_vector_multiplier<1 | HRscale_vector_multiplier>1)) > 0) {
		cat("**************************************************************************")
		cat("WARNING - one or more baseline Harvest Ratio scaling values differs from 1")
		cat("**************************************************************************")
	}

	#Set up a dataframe to hold the proposed and accepted HR_scale_vector

	HRscale_proposal<-data.frame(PF=0,DF=0,MF=0,Bsd=0,Bcs=0,Zc=0,BD=0,SL=0,CT=0,KP=0,lik=0)
	HRscale_proposal[1,]<-c(HRscale_vector_multiplier, 1e-60)
	HRscale_accepted<-HRscale_proposal

	n_acceptances <- 0	# Counter for the number of parameter acceptances which have occurred
	lastbest <- 1e-60

	temperature<-start_temperature

	#ITERATE THE ANNEALING PROCESS.....

	for (kkk in 1:n_iter){

		annealing_control_data <- read_SD_fishing(model.path)

			if(kkk==1){
			axmin <- elt(annealing_control_data, "axmin")
			axmax <- elt(annealing_control_data, "axmax")
			axmin_p<-axmin
			axmax_p<-axmax
			newglims<-1
			}
			if(kkk>1){
			newglims<-0
			axmin <- elt(annealing_control_data, "axmin")
				if((axmin==axmin_p)==FALSE){
					newglims<-1
				}
			axmax <- elt(annealing_control_data, "axmax")
				if((axmax==axmax_p)==FALSE){
					newglims<-1
				}
			axmin_p<-axmin
			axmax_p<-axmax
			}

		if( axmax<=axmin ) {
		cat("Warning: Plot axis max <= axis min in optimize_fishing.csv parameter file - reset to defaults","\n")
		axmax<-1
		axmin<-0
		newglims<-1
		}

		deltaH <- elt(annealing_control_data, "deltaH")		# cv for jiggling the fishing fleet parameters
		#Values for the cv are held in the "optimize_fishing" file.
		#THESE CAN BE ALTERED DURING A RUN BY MANUALLY EDITING VALUES IN THE CONTROL FILE
		# Suggested values are:
		#    deltaH<-0.10  # high value for the first cycle through all this
		#    deltaH<-0.05  # medium value is strating from something half sensible
		#    deltaH<-0.02  # small vakue for the second cycle through all this
		#    deltaH<-0.01  # very small value if strating from something close

		if(kkk>1){
			for(www in 1:10){
				HRscale_vector_multiplier[www] <- max(0,rnorm(1,HRscale_accepted[(kkk-1),www],deltaH*HRscale_accepted[(kkk-1),www]))
				#NON-STANDARD OPTION that might be needed - force the MF HR scaler to be same as PF HR scaler
				#if(www==3) HRscale_vector_multiplier[3]<-HRscale_vector_multiplier[1]
			}
			HRscale_proposal[kkk,]<-c(HRscale_vector_multiplier,0)
			HRscale_accepted[kkk,]<-HRscale_accepted[(kkk-1),]

			model$data$fleet.model$HRscale_vector_multiplier <- HRscale_vector_multiplier
		}
		# showall("HRscale_vector_multiplier", HRscale_vector_multiplier)

		# Build and run the model:
		build <- build_model(model, nyears)
		fit <- StrathE2E_fit(model, build, quiet)
		annual_obj <- elt(fit, "annual_obj")

		# cat("Err=", annual_obj, "\n")

		HRscale_proposal[kkk,11]<-annual_obj

		temperature<-temperature*cooling

		#-------------------------------------------------------------------
		#Now the Metropolis algorithm.....

		lik_ratio<-exp(((log(annual_obj)) - log(lastbest))/temperature)
		rand<-runif(1,0,1)

		new_accepted<-" accepted: NO"
		if(lik_ratio>rand){
			n_acceptances<-n_acceptances+1
			HRscale_accepted[kkk,]<-HRscale_proposal[(kkk),]
			lastbest<-annual_obj

			new_accepted<-" accepted: YES"
		}

		#--------------------------------

		filename = csvname(resultsdir, "annealing_HRmult_proposalhistory", identifier)
		writecsv(HRscale_proposal, filename, row.names=FALSE)

		filename = csvname(resultsdir, "annealing_HRmult_acceptedhistory", identifier)
		writecsv(HRscale_accepted, filename, row.names=FALSE)

		#-------------------------------------------------------------------

		cat("Iteration: ",kkk,"; proposal likelihood: ",sprintf("%.7f",HRscale_proposal[kkk,11]),"; ",new_accepted,"\n",sep="")


#----------------------------------------------------------------------------------------------------------

		#Plot or update the time series of proposal and acepted likelihoods so far....
		if(runtime.plot==TRUE){
		if(kkk>1){

			if(kkk==2 | newglims==1){
			par(mfrow=c(1,1))
			plot(seq(1:nrow(HRscale_proposal)),HRscale_proposal[,11],ylim=c(axmin,axmax),xlim=c(1,n_iter),xlab="Iterations",ylab="Target data likelihood",type="l",col="grey")
			points(seq(1,nrow(HRscale_accepted)),HRscale_accepted[,11],type="l",col="black",lwd=3)
                	legend("topleft",c("accepted","proposed"),bg="transparent",col=c("black","grey"),lty=c(1,1),lwd=c(3,1),pt.cex=c(1,1))
			}

			if(newglims==0){
			stp<-c((kkk-1),kkk)
			points(stp,HRscale_proposal[stp,11],type="l",col="grey")
			points(stp,HRscale_accepted[stp,11],type="l",col="black",lwd=3)
			} 

		}
		}

	}

#----------------------------------------------------------------------------------------------------------

	new_parameters<-extract_HRmult_to_parmsfolder(model, HRscale_accepted,csv.output=csv.output)

	#Assemble a list object containing all the data generated by the annealing process
	output_data<-list(parameter_proposal_history = HRscale_proposal,
			  parameter_accepted_history = HRscale_accepted,
			  new_parameter_data = new_parameters)
	output_data

}

