#
# e2ep_calculate_hrscale.R
#
#' Calculate initial values of the scaling parameters for the fishing fleet model which link effort to harvest ratios.
#'
#' The scaling parameters convert the effort applied by each gear to each living resource guild in the model, into a harvest ratio (proportion of mass captured per day).
#' Effort is the product of activity and the relative power of each gear with respect to each guild.
#'
#' In order to estimate these scaling parameters, data are needed for a 'calibration' time period when activity, catch and harvest ratio are all known.
#'
#' The function assumes that the relevant model configuration has already been loaded using the e2ep_read() function. It is expected that
#' this has loaded the array of power parameters (catch per unit activity by gear and guild during the calibration period) given in the file /Param/fishing_power_*.csv,
#' the corresponding activity in the file /Param/fishing_activity_*.csv, and the known harvest ratios of each guild in the file /Target/region_harvest_r_*.csv
#'
#' The function returns rough estimates of the scaling parameters. These are only rough because they assume that the effort and resource biomass are distributed uniformly
#' over the model domain. Hence they should be used as initial estimates to be refined by optimization.
#'
#' The scaling parameters are dispayed on the screen and saved in a dataframe. The values then need to be manually edited
#' into rows 12-21 (excluding the header row) of the file /Param/fishing_fleet_*.csv
#'
#' @param model R-list object generated by the e2ep_read() function which defined the model configuration
#'
#' @return Dataframe of the effort-harvest ratio scaling parameter values for each guild
#'
#' @seealso \code{\link{e2ep_ls}} , \code{\link{e2ep_read}}
#'
#' @export
#'
#' @examples
#' # Load the 2011-2019 version of the Barents Sea model supplied with the package and calculate
#' # scaling parameter values:
#'     model <- e2ep_read("Barents_Sea", "2011-2019", quiet=FALSE)
#'     scale_values <- e2ep_calculate_hrscale(model)
#'     scale_values
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------


e2ep_calculate_hrscale <- function(model) {

   oo <- options()
   on.exit(options(oo))

	model.path			<- elt(model, "setup", "model.path")
	model.ident			<- elt(model, "setup", "model.ident")

	data				<- elt(model, "data")
	fleet.model			<- elt(data, "fleet.model")
	gear_mult			<- elt(fleet.model, "gear_mult")
	gear_activity			<- elt(fleet.model, "gear_activity")
	gear_group_rel_power		<- elt(fleet.model, "gear_group_rel_power")
	HRscale_vector_multiplier	<- elt(fleet.model, "HRscale_vector_multiplier")

	known_HRs <- get.model.file(model.path, TARGET_DATA_DIR, file.pattern=KNOWN_HARVEST_RATIO_VALUES)

	if(length(which(HRscale_vector_multiplier<1|HRscale_vector_multiplier>1))>0) {
		message("**************************************************************************")
		message("WARNING - one or more baseline Harvest Ratio scaling values differs from 1")
		message("**************************************************************************")
	}

	# Apply the group fishing multiplers - - these are activity scaling factors used to generate scenarios of fishing activity
	# The scaling factors are applied across groups of gears which mainly target the same resource groups in the model
	#
	gear_activity_mult <- gear_activity * gear_mult
	gear_group_effort <- gear_activity_mult*gear_group_rel_power

	# Calculate scaling coefficient between regional average daily effort density and daily harvest ratio
	#
	HRscale_vector <-(rep(0,10))
	for(kkk in 1:10){
		if(known_HRs[kkk,2]>0) {
			HRscale_vector[kkk] <- known_HRs[kkk,2]/sum(gear_group_effort[,kkk])
		}
	}
	HRscale_vector_df<-data.frame("Guild"=known_HRs[,1],"HRscale"=HRscale_vector)
	HRscale_vector_df

	message("Manually edit these values into the 'fishing_fleet_*.csv' file, rows 12-21, in the 'Param' folder")

	HRscale_vector_df
}

