#
# plot_time_series_annual_land_disc.R
#
#' Time series plots of annual integrated catches from the inshore and offshore zones
#'
#' Multi-panel time series bar-plots of annual integrated catch of each living guild in the inshore and offshore zones over the full length of a model run generated by the e2ep_run() function
#'
#' The function plots a multi-panel page of time series bar-plots of annual integrated catches
#' of the potentially catchable guilds in the ecology model:
#' macrophytes, carnivorous zooplankton, suspension/deposit and carnivore/scavenge feeding benthos, planktivorous, demersal and migratory fish,
#' birds, pinnipeds and cetaceans.
#'
#' Each panel represents a single guild. In each plot panel, the x-axis represents the sequence of years of the model run, and the y-axis is the annual integrated catch.
#' For each year there are two bars corresponding to the inshore and offshore zones. Each bar is subdivided into landed and discarded components ofthe catch.
#' Green and blue sections are offshore and inshore landings respectively; black and grey sections are offshore and inshore discards respectively.
#'
#' Units for the catch are mMN/year from the whole model domain (1 m2) - i.e. not scaled to zonal area.
#'
#' @param model R-list object defining the model configuration compiled by the e2ep_read() function
#' @param results R-list object containing model results generated by the e2ep_run() function
#'
#' @return Graphical display in a new graphics window
#'
#' @importFrom graphics barplot title grconvertX grconvertY
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_time_series_annual_land_disc <- function(model, results) {
    
start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	#----------------------------------------------------------------
	#In each plot panel:
	#    black and grey are offshore and inshore discards respectively
	#    green and blue are offshore and inshore landings respectively
	#----------------------------------------------------------------

	total.annual.catch <- elt(results, "total.annual.catch")

        ngroups<-10 # this is the number of groups in gear_group_proportions


        offshore_annual_group_land_disc <- elt(total.annual.catch, "offshore_annual_group_land_disc")
        inshore_annual_group_land_disc  <- elt(total.annual.catch, "inshore_annual_group_land_disc")

	group_names <- c(
		"Planktivorous fish",
		"Quota-limited Demersal fish",
		"Non-quota demersal fish",
		"Migratory fish",
		"Susp/dep feeding benthos",
		"Carn/scav feeding benthos",
		"Pelagic invertebrates",
		"Birds",
		"Pinnipeds",
		"Cetaceans","Macrophytes"
	)

	par(mfrow=c(6,2))
	par(mar=c(4,5,1,0.5))


	for(dsa in 1:(ngroups+1)) {

		mt<-group_names[dsa]

		offshore_data2plot<-rbind(offshore_annual_group_land_disc[,(ngroups+2)+dsa],offshore_annual_group_land_disc[,1+dsa])
		inshore_data2plot<-rbind(inshore_annual_group_land_disc[,(ngroups+2)+dsa],inshore_annual_group_land_disc[,1+dsa])

		offshore_catch<-colSums(offshore_data2plot)
		inshore_catch<-colSums(inshore_data2plot)

		colnames(offshore_data2plot)<-offshore_annual_group_land_disc[,1]

		barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,0.0000001+(1.2*max(c(offshore_catch,inshore_catch)))),xlim=c(0,(ncol(offshore_data2plot)+1)),width=rep(0.4,ncol(offshore_data2plot)), space=c(0.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",ann=FALSE,cex.axis=0.9)
		axis(side=2,las=1,cex.axis=0.9)
		if(dsa<11)  mtext("Years",cex=0.7,side=1,line=2)
		mtext("Catch",cex=0.7,side=2,line=3.5)
		title(main=mt,cex.main=1.0)
		barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,ncol(offshore_data2plot)), space=c(1.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",xaxt="n",ann=FALSE)
	}


	#Calculate total landings and disards avross all guilds
	offshore_data2plot<-rbind(rowSums(offshore_annual_group_land_disc[,((ngroups+2)+1) : ((ngroups+2)+(ngroups+1))]),rowSums(offshore_annual_group_land_disc[,2:(ngroups+2)]))
	inshore_data2plot<- rbind(rowSums( inshore_annual_group_land_disc[,((ngroups+2)+1) : ((ngroups+2)+(ngroups+1))]),rowSums( inshore_annual_group_land_disc[,2:(ngroups+2)]))

	offshore_catch<-colSums(offshore_data2plot)
	inshore_catch<-colSums(inshore_data2plot)

	colnames(offshore_data2plot)<-offshore_annual_group_land_disc[,1]

	mt<-"All guilds combined"

	barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,0.0000001+(1.2*max(c(offshore_catch,inshore_catch)))),xlim=c(0,(ncol(offshore_data2plot)+1)),width=rep(0.4,ncol(offshore_data2plot)), space=c(0.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",ann=FALSE,cex.axis=0.9)
	axis(side=2,las=1,cex.axis=0.9)
	mtext("Catch",cex=0.7,side=2,line=3.5)
	title(main=mt,cex.main=1.0)
	barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,ncol(offshore_data2plot)), space=c(1.5,rep(1.2,ncol(offshore_data2plot)-1)),yaxt="n",xaxt="n",ann=FALSE)

	legend(grconvertX(0.15, "ndc", "user"), grconvertY(0.05, "ndc", "user"),
	c( "offshore landings","offshore discards","inshore landings","inshore discards"),
		fill = c("green","black","blue","grey"), ncol=4, bty="n", xpd = NA)

}

#----------------------------------------------------------------------------------------
