#
# e2ep_compare_obs.R
#
#' Box and whisker plots comparing annual or monthly observational data with corresponding model outputs.
#'
#' Generate a multi-panel set of box and whisker diagrams comparing annual or monthly averaged or integrated observational data on the state of the
#' ecosystem with corresponding model outputs. Each panel displays a different category of observational and model data. 
#'
#' The observational data are read from the file 'annual_observed_*.csv' or 'monthly_observed_*.csv'in the /Target folder of the model setup defined by a e2ep_read() function call.
#' Column 3 of annual_observed_* (header "Use_1.yes_0.no") is a flag to set whether any given row is used in calculating the likelihood of the observed data given the model setup by functions such 
#' as e2ep_optimize_eco(). Un-used rows of data are omitted from the box and whisker plotting panels.
#'
#' Arguments determine the source of model data for comparison with the observations. These can be outputs from a single model run with data held in memory as a list object or in a saved csv file, or from 
#' a Monte Carlo simulation (using the function e2ep_run_mc()) to estimate credible intervals of model outputs. Generation of credible interval data is a long computing task, so 
#' example data for the Barents Sea model provided with the package for illustration.
#' 
#' In each plot panel, the distribution of observation data is shown by a black box plot (box spans 50% of obervations, whiskers span 99%, median indicated by a black tick mark). Corresponding data 
#' derived from model outputs are always shown in red - either a red tick mark for data from a single model run, or a red boxplot for data from a Monte Carlo analysis.
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2ep_read() function.
#' @param selection Text string from a list to select comparison with annual or monthly observations. Select from: "ANNUAL", "MONTHLY". Remember to include the phrase within "" quotes.
#' @param ci.data Logical. If TRUE plot credible intervals around model results based on Monte Carlo simulation with the e2ep_run_mc() function (default=FALSE).
#' @param use.saved Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).
#' @param use.example NOT YET ENABLED. In future versions - Logical. If TRUE use pre-computed example data from the internal Barents Sea model rather than user-generated data (default=FALSE).
#' @param results R-list object of model output generated by the e2ep_run() function (default=NULL).
#'
#' @return graphical display in a new graphics window
#'
#' @seealso \code{\link{e2ep_read}}, \code{\link{e2ep_run}}, \code{\link{e2ep_run_mc}}
#'
#' @importFrom graphics text
#'
#' @export
#'
#' @examples
#' # Load the 2011-2019 version of the Barents Sea model supplied with the package, run,
#' # and generate a plot:
#'     model <- e2ep_read("Barents_Sea", "2011-2019")
#'     results <- e2ep_run(model, nyears=2,csv.output=FALSE)
#' # Plot data annual data
#'     e2ep_compare_obs(selection="ANNUAL", model, results=results)
#' # Note that these are the observational data that were used as the target for optimizing
#' # the model parameters
#'
#' # Direct the graphics output to a file ... in this example the graphics file
#' # is sent to a temporary folder rather than the current working directory:
#' # or jpeg("plot.jpg"), png("plot.png")
#'     pdf(file.path(tempdir(), "plot.pdf"),width=8,height=6)
#'     e2ep_compare_obs(selection="ANNUAL", model, results=results)
#'     dev.off()
#'
#' # Plot monthly data
#'     dev.new()
#'     e2ep_compare_obs(model, selection="MONTHLY",results=results)
#' # Note that these observational data were NOT used for optimizing the 
#' # model parameters
#'
#' \donttest{
#' # To create the same plots from the csv files saved by the e2ep_run() function, use:
#'     model <- e2ep_read("Barents_Sea", "2011-2019")
#'     results <- e2ep_run(model, nyears=2,csv.output=TRUE)
#' # Here the csv outputs are saved to a temporary folder since results.path is not 
#' # set in e2ep_read()
#'     e2ep_compare_obs(selection="ANNUAL", model, use.saved=TRUE)
#'     dev.new()
#'     e2ep_compare_obs(selection="MONTHLY", model, use.saved=TRUE)
#' }
#'
#' # Load the 2011-2019 version of the Barents Sea model supplied with the package and plot
#' # example data with credible intervals generated by a baseline mode Monte Carlo analysis.
#' # This example requires the StrathE2EPolarexamples supplementary data package.
#' if(require(StrathE2EPolarexamples)){
#'     model <- e2ep_read("Barents_Sea", "2011-2019")
#'     e2ep_compare_obs(model, selection="ANNUAL",ci.data=TRUE,use.example=TRUE)
#'     dev.new()
#'     e2ep_compare_obs(selection="MONTHLY", model, ci.data=TRUE,use.example=TRUE)
#' }
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

e2ep_compare_obs <- function(selection="ANNUAL", model, ci.data=FALSE, use.saved=FALSE, use.example=FALSE, results=NULL  ) {

   oo <- options()
   on.exit(options(oo))

######################################
if(use.example==TRUE){
        stop("use.example not enabled in this version of StrathE2EPolar ! \n")
}
######################################

	if (use.example == TRUE) hasExamples()

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

if(use.saved==TRUE & use.example==TRUE){
        stop("use.saved and use.example cannot both be TRUE ! \n")
}

if(use.saved==TRUE & is.list(results)==TRUE){
        stop("use.saved is TRUE but a dataframe object has also been specified ! \n")
}

if(use.example==TRUE & is.list(results)==TRUE){
        stop("use.example is TRUE but a dataframe object has also been specified ! \n")
}

if(use.saved==FALSE & use.example==FALSE & is.list(results)==FALSE){
        stop("no source of data has been specified ! \n")
}

if(ci.data==FALSE & use.example==TRUE){
        stop("no example data available for a single model run - run the model to generate some data ! \n")
}

if(ci.data==TRUE & use.saved==FALSE & use.example==FALSE & is.list(results)==TRUE){
        stop("credible intervals available only from saved or example data ! \n")
}

#.........................................................................

if(selection=="ANNUAL"){

if(ci.data == FALSE & use.saved==FALSE & is.list(results)==TRUE ){
	boxplot_annual_compare_observations(model, use.saved=FALSE,results=results)
}

if(ci.data == FALSE & use.saved==TRUE ){
	boxplot_annual_compare_observations(model, use.saved=TRUE, results=NULL)
}

if(ci.data == TRUE & use.saved==FALSE & use.example==TRUE ){
	boxplot_annual_compare_observations_with_ci(model, use.example=TRUE)
}

if(ci.data == TRUE & use.saved==TRUE & use.example==FALSE ){
	boxplot_annual_compare_observations_with_ci(model, use.example=FALSE)
}


#.........................................................................

} else if(selection=="MONTHLY"){

if(ci.data == FALSE & use.saved==FALSE & is.list(results)==TRUE ){
	boxplot_monthly_compare_observations(model, use.saved=FALSE,results=results)
}

if(ci.data == FALSE & use.saved==TRUE ){
	boxplot_monthly_compare_observations(model, use.saved=TRUE)
}

if(ci.data == TRUE & use.saved==FALSE & use.example==TRUE ){
	boxplot_monthly_compare_observations_with_ci(model, use.example=TRUE)
}

if(ci.data == TRUE & use.saved==TRUE & use.example==FALSE ){
	boxplot_monthly_compare_observations_with_ci(model, use.example=FALSE)
}


#.........................................................................

} else {
	stop("Error: unknown selection '", selection, "' !\n")
}

#.........................................................................

}

#-----------------------------------------------------------------


