#
# e2ep_extract_hr.R
#
#' Extract the values of harvest ratios generated by the fleet model.
#'
#' The function extracts the inshore, offshore and whole-domain values of harvest ratio (proportion of biomass captured per day) for each guild, from the results object generated by the e2e_run() function.
#'
#' @param model R-list object defining the model configuration compiled by the e2e_read() function.
#' @param results R-list object generated by the e2e_run() function.
#' @param csv.output Logical. If TRUE then enable writing of CSV output files (default=FALSE).
#'
#' @return Dataframe of harvest ratio values, rows = guilds, columns = zone
#'
#' @seealso \code{\link{e2ep_read}}, \code{\link{e2ep_run}} 
#'
#' @export
#'
#' @examples
#'     model <- e2ep_read("Barents_Sea", "2011-2019") 
#'     results<-e2ep_run(model, nyears=2,csv.output=FALSE)
#'     harvest_ratio_data <- e2ep_extract_hr(model,results,csv.output=FALSE)
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------


e2ep_extract_hr <- function(model, results,csv.output=FALSE) {

   oo <- options()
   on.exit(options(oo))

	pkg.env$csv.output <- csv.output	# controls writing of CSV files

	setup		<- elt(model, "setup")
	model.path	<- elt(setup, "model.path")
	resultsdir	<- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")

	model.parameters<- elt(results,"build","model.parameters")
	fleet_vector	<- elt(results,"fleet.output","fleet_vector")

	grouplabels <- c(
		"Planktivorous_fish",
		"Demersal_fish",
		"Migratory_fish",
		"Benthos_susp-dep",
		"Benthos_carn-scav",
		"Zooplankton_carn",
		"Birds",
		"Pinnipeds",
		"Cetaceans",
		"Macrophytes"
	)

	shallowprop<-model.parameters["shallowprop"]

	zonalHRs<-data.frame(Guilds=grouplabels,Inshore_harvest_ratio=rep(0,length(grouplabels)),Offshore_harvest_ratio=rep(0,length(grouplabels)),Whole_domain_harvest_ratio=rep(0,length(grouplabels)))

	zonalHRs$Inshore_harvest_ratio<-c(fleet_vector["F_inshore_pelagic"],
				 fleet_vector["F_inshore_demersal"],
				 fleet_vector["F_inshore_migratory"],
				 fleet_vector["F_inshore_filtben"],
				 fleet_vector["F_inshore_carnben"],
				 fleet_vector["F_inshore_carnzoo"],
				 fleet_vector["F_inshore_bird"],
				 fleet_vector["F_inshore_seal"],
				 fleet_vector["F_inshore_ceta"],
				 fleet_vector["F_inshore_kelp"])

	zonalHRs$Offshore_harvest_ratio<-c(fleet_vector["F_offshore_pelagic"],
				 fleet_vector["F_offshore_demersal"],
				 fleet_vector["F_offshore_migratory"],
				 fleet_vector["F_offshore_filtben"],
				 fleet_vector["F_offshore_carnben"],
				 fleet_vector["F_offshore_carnzoo"],
				 fleet_vector["F_offshore_bird"],
				 fleet_vector["F_offshore_seal"],
				 fleet_vector["F_offshore_ceta"],
				 fleet_vector["F_offshore_kelp"])

	zonalHRs$Whole_domain_harvest_ratio<- (zonalHRs$Inshore_harvest_ratio*shallowprop) + (zonalHRs$Offshore_harvest_ratio*(1-shallowprop))

		filename = csvname(resultsdir, "harvest_ratios_by_guild", model.ident)
		writecsv(zonalHRs, filename, row.names=FALSE)

	return(zonalHRs)

}
