#
# e2ep_get_parmdoc.R
#
#' Get documentation on the parameters in the model.
#'
#' Provides access to documentation on the parameters in the model in the form of a dataframe.
#'
#' There are two levels of parameters for the model - the raw input parameters as extracted from the parameter csv files, and an expanded set of parameters, derive from the raw values, which are injected into the the model run.
#' 
#' The Monte Carlo simulation function in the package acts on the raw ecology model parameters, jigging each by a a random increment, to generate test parameter sets. The ecology model parameters are divided into fixed and fitted subsets based on whether they are available for optimization or not.
#' 
#' The Global Sensitivity analysis function in the package acts on the expanded parameter set. As well as fixed and fitted ecology parameters, the expanded set includes the physical setup parameters, fishing fleet parameters, harvest ratios, and environmental drivers.
#'
#' This function downloads documentation on the full range of parameters in the form of a dataframe which can be used to form labels in user-generated code. The parameters are identified by a numeric value linked to 6 different classes of parameter.
#'
#' Note that the function does not supply the numeric values of parameters for any given model setup. These are available in the model definition object generated by the function e2e_read().
#'
#' @param selection Text string from a list identifying the level of parameters to be listed. Select from: "MC", "SENS", Remember to include the phrase within "" quotes (default = "SENS")
#' @param id Integer value denoting the class of parameters details to be downloaded. Choose from: 0 = fitted ecology, 1 = fixed ecology, 2 = fishing fleet, 3 = harvest ratios, 4 = environmental drivers, 5 = physical configuration, 9 = all (default = 9). Options 2-5 not available with "MC".
#'
#' @return dataframe of parameter documentation.
#'
#' @seealso \code{\link{e2ep_read}}
#'
#' @export
#'
#' @examples
#' # Details of all the fitted ecology parameters used in the Monte Carlo function:
#'     parm_list <- e2ep_get_parmdoc("MC",0)        
#' # Details of all parameters in the sensitivity analysis function:
#'     parm_list <- e2ep_get_parmdoc("SENS",9)
#' # ... or simply:
#'     parm_list <- e2ep_get_parmdoc()
#'
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: July 2021                                   |
# |                                                                   |
# ---------------------------------------------------------------------

e2ep_get_parmdoc <- function(selection="SENS",id=9) {

   oo <- options()
   on.exit(options(oo))

allowed.list.sel  <- c("MC","SENS")
allowed.list.mc   <- c(0,1,2,9)
allowed.list.sens <- c(0,1,2,3,4,5,9)

if(selection %in% allowed.list.sel==FALSE){
	message("The argument 'selection' must be one of the following...")
	message("    MC = raw parameters used in the Monte Carlo scheme")
	message("    SENS = extended parameters used in the Global Sensisitivity analysis scheme")
	message("    unrecognised value entered - reverting to the default")
        selection<-"SENS"
}

if(selection=="SENS"){
if(id %in% allowed.list.sens==FALSE){
	message("The argument 'id' must be one of the following...")
	message("    0 = fitted ecology")
	message("    1 = fixed ecology")
	message("    2 = fishing fleet")
	message("    3 = harvest ratios")
	message("    4 = environmental drivers")
	message("    5 = physical configuration")
	message("    9 = all parameters (default)")
	message("    unrecognised value entered - reverting to the default")
        id<-9
}
parameter.details.data.file <- system.file("extdata/Internal.data", "StrathE2EPolar_sens_parameter_list.csv", package="StrathE2EPolar", mustWork = TRUE)
}

if(selection=="MC"){
if(id %in% allowed.list.mc==FALSE){
	message("The argument 'id' must be one of the following...")
	message("    0 = fitted ecology")
	message("    1 = fixed ecology")
	message("    9 = all parameters (default)")
	message("    unrecognised value entered - reverting to the default")
        id<-9
}
parameter.details.data.file <- system.file("extdata/Internal.data", "StrathE2EPolar_mc_parameter_list.csv", package="StrathE2EPolar", mustWork = TRUE)
}

parameter.details <- read.csv(parameter.details.data.file,header=TRUE)

	
if(id==9) selected_class<-parameter.details

if(id < 9) selected_class<-parameter.details[which(parameter.details$fixfit==id),]

return(selected_class)

}



