#
# e2ep_plot_eco.R
#
#' Plot daily data on ecological outputs from the model over the final year of a run, optionally with credible intervals.
#'
#' Generate a multi-panel set of one-year time series plots of selected outputs on the concentrations of state variables in the ecology model. The default is to plot data from a single model run but if available, 
#' credible intervals of model output from a Monte Carlo analysis can be plotted instead.
#'
#' Arguments determine the source of model data to be plotted. These can be outputs from a single model run with data held in memory as a list object or in a saved csv file, or from 
#' a Monte Carlo simulation (using the function e2ep_run_mc()) to estimate credible intervals of model outputs. Generation of credible interval data is a long computing task, so 
#' example data for the Barents Sea model provided with the package are available as an illustration.
#'
#' If plotting of credible intervals is selected, results from the maximum likelihood model are shown by a red line. The median of the credible values distribution
#' is shown my a solid black line. A grey-shaded area indicates the 50% credible interval (spanning quartiles of the cumulative likelihood
#' of simulated values). Black dashed lines span the 99% credible interval.
#'
#' Variables to be plotted (and units) depending on values of "selection":
#' \itemize{
#'   \item NUT_PHYT    : Water column nitrate, ammonia, detritus and phytoplankton             (mMN/m3)
#'   \item IN_ICE      : In-snow/ice concentrations of nitrate, ammonia, detritus and algae    (mMN/m3 or mMN/m2 of snow or ice)
#'   \item ZONE_ICE    : Zonal area densities of snow/ice nitrate, ammonia, detritus and algae (mMN/m2 of zone)
#'   \item SEDIMENT    : Sediment porewater nitrate and ammonia, detritus and corpses          (mMN/m3 or gN/gDW)
#'   \item ZOOPLANKTON : Omnivorous and carnivorous zooplankton                                (mMN/m2)
#'   \item FISH        : Planktivorous and demersal fish and fish larvae                       (mMN/m2)
#'   \item BENTHOS     : Susp/dep and carn/scav benthos and benthos larvae                     (mMN/m2)
#'   \item PREDATORS   : Birds, pinnipeds, cetaceans and migratory fish                        (mMN/m2)
#'   \item CORP_DISC   : Corpses and discards                                                  (mMN/m2)
#'   \item MACROPHYTE  : Macrphytes and macrophyte debris                                      (mMN/m2)
#' }
#'
#' To direct the graph output to a file rather than the screen, wrap the e2ep_plot_eco() function call in a graphical device call:
#' Since the plot pages contain different numbers of panels the suggested width:height ratios are as follows:
#' \itemize{
#'   \item NUT_PHYT .... 1.5 : 1
#'   \item IN_ICE ...... 1.5 : 1
#'   \item ZONE_ICE .... 1.5 : 1
#'   \item SEDIMENT ... 0.67 : 1
#'   \item ZOOPLANKTON ... 1 : 1
#'   \item FISH .......... 2 : 1
#'   \item BENTHOS ....... 2 : 1
#'   \item PREDATORS ..... 2 : 1
#'   \item CORP_DISC ..... 1 : 1
#'   \item MACROPHYTE .... 2 : 1
#' }
#'
#' @param model R-list object defining the baseline model configuration used to generate the data and compiled by the e2ep_read() function.
#' @param selection Text string from a list identifying the group of model output variables to be plotted. Select from: "NUT_PHYT", "SEDIMENT", "ZOOPLANKTON", "FISH", "BENTHOS", "PREDATORS", "CORP_DISC", "MACROPHYTE", Remember to include the phrase within "" quotes.
#' @param ci.data Logical. If TRUE plot credible intervals around model results based on Monte Carlo simulation with the e2ep_run_mc() function (default=FALSE).
#' @param use.saved Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).
#' @param use.example NOT YET ENABLED. In future versions - Logical. If TRUE use pre-computed example data from the internal Barents Sea model rather than user-generated data (default=FALSE).
#' @param results R-list object of model output generated by the e2ep_run() function. Only needed if ci.data=FALSE, use.saved=FALSE and use.example=FALSE. (Default=NULL).
#'
#' @return Graphical display in a new graphics window.
#'
#' @seealso \code{\link{e2ep_read}}, \code{\link{e2ep_run}}, \code{\link{e2ep_plot_migration}}, \code{\link{e2ep_plot_catch}}, \code{\link{e2ep_plot_trophic}}, \code{\link{e2ep_plot_biomass}}
#'
#' @importFrom graphics text
#'
#' @export
#'
#' @examples
#' # Load the 2011-2019 version of the Barents Sea model supplied with the package, run, and
#' # generate a plot:
#'     model <- e2ep_read("Barents_Sea", "2011-2019")
#'     results <- e2ep_run(model, nyears=2,csv.output=FALSE)
#'     e2ep_plot_eco(model, selection="NUT_PHYT",results=results)
#'     dev.new()
#'     e2ep_plot_eco(model, selection="ZOOPLANKTON",results=results)
#'
#' # Direct the graphics output to a pdf file ... 
#' # or jpeg("plot.jpg"), png("plot.png")
#'     pdf(file.path(tempdir(), "plot.pdf"),width=8,height=4)
#'     e2ep_plot_eco(model, selection="FISH",results=results)
#'     dev.off()
#'
#' # Load the 2011-2019 version of the Barents Sea model supplied with the package and
#' # plot example credible interval data:
#' # This example requires the StrathE2EPolarexamples supplementary data package.
#' if(require(StrathE2EPolarexamples)){
#'     e2ep_plot_eco(model, selection="BENTHOS",ci.data=TRUE,use.example=TRUE)
#' }
#'
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Author: Mike Heath                                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

e2ep_plot_eco <- function(model, selection="NUT_PHYT", ci.data=FALSE, use.saved=FALSE, use.example=FALSE, results=NULL ) {

   oo <- options()
   on.exit(options(oo))

######################################
if(use.example==TRUE){
        stop("use.example not enabled in this version of StrathE2EPolar ! \n")
}
######################################

	if (use.example == TRUE) hasExamples()

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

if(use.saved==TRUE & use.example==TRUE){
        stop("use.saved and use.example cannot both be TRUE ! \n")
}

if(use.saved==TRUE & is.list(results)==TRUE){
        stop("use.saved is TRUE but a results list object has also been specified ! \n")
}

if(use.example==TRUE & is.list(results)==TRUE){
        stop("use.example is TRUE but a dataframe object has also been specified ! \n")
}

if(use.saved==FALSE & use.example==FALSE & is.list(results)==FALSE){
        stop("no source of data has been specified ! \n")
}

if(ci.data==FALSE & use.example==TRUE){
        stop("no example data available for a single model run - run the model to generate some data ! \n")
}

if(ci.data==FALSE & use.saved==TRUE){
        stop("no daily data files available for a single model run - run the model to generate some data ! \n")
}

if(ci.data==TRUE & use.saved==FALSE & use.example==FALSE & is.list(results)==TRUE){
        stop("credible intervals available only from saved or example data ! \n")
}

#.........................................................................

if(ci.data == FALSE & use.saved==FALSE & is.list(results)==TRUE ){
	plot_final_year_time_series_data(model=model, results=results, selection=selection)

} else if(ci.data == TRUE & use.saved==FALSE & use.example==TRUE ){
	plot_final_year_time_series_data_with_ci(model=model, selection=selection, use.example=use.example)

} else if(ci.data == TRUE & use.saved==TRUE & use.example==FALSE ){
	plot_final_year_time_series_data_with_ci(model=model, selection=selection, use.example=use.example)

#.........................................................................

} else {
	stop("Error: mistake in argument settings !","\n")
}

#.........................................................................

}

#-----------------------------------------------------------------


