#
# plot_mc_parameter_likelihoods.R
#
#' Plots the credible distributions of ecology model parameters from a Monte Carlo simulation.
#'
#' Create a plot showing the credible distributions of ecology model parameters based on the results from the e2ep_run_mc() function.
#' These distributions are formed from the input distributions to the Monte Carlo process, weighted by the likelihood of observed target data
#' on the state of the ecosystem given each combination of parameter values.
#' 
#' There are two classes of ecology model parameters. The 'fitted' parameters that are subject to optimization by the e2ep_optimize_eco() function, and those 
#' that are 'fixed' and not subject to optimization. The Monte Carlo scheme does not distinguish between these two classes - each iteration draws random
#' values of both classes.
#'
#' Post-processed data from the e2ep_run_mc() function are stored in the file
#' /results/Modelname/Variantname/CredInt/CredInt_processed_PARAMETERresults_data-*.csv, where * represents the model run identifier
#' (model.ident) text embedded in the R-list object created by the e2ep_read() function.
#'
#' Optionally the function can read an example data set for one of the two North Sea model variants supplied with the package.
#'
#' Each parameter in the plot is scaled to its baseline value as (value - baseline)/baseline. Ideally, the baseline is the maximum likelihood model
#' developed by application of the optimization functions available in the package (e.g. e2ep_optimize_eco()).
#' Each parameter is then represented by a box and whisker plot which shows the distribution of credible parameter values around zero, i.e.
#' around the baseline. The distributions of 'fitted' parameters are shown by blue box and whiskers, the 'fixed' parameters by black. 
#' Each box spans the 50% credible interval (quartiles of the cumulative likelihood
#' of simulated values). Whisker lines span the 99% credible interval.
#'
#' The individual parameters are identified by numbers (rather than text names). These numnbers correspond to the column numnbers in
#' the file /results/Modelname/Variantname/CredInt/CredInt_processed_parameters-*.csv . Details of the parameters associated with each identification
#' number are available as a dataframe by using the function e2ep_get_parmdoc().
#' 
#' The input distribution of parameter values to the Monte Carlo process is drawn from a random uniform distribution with limits specified in
#' the monte_carlo control file for the model setup (located in a sub-folder of the /Param/control folder). This distribution is shown by
#' a red box and whisker at the bottom of the plot. Given the random uniform input we expect the quartiles (limits of the box) to be symetrical
#' and located mid-way between zero and the upper and lower extremes. Vertical red lines show the expected limits of the quartiles boxes if 
#' model results were completely insensitive to individual parameter values.
#'
#' The extent to which individual parameter distributions deviate from the random uniform input is an indication of their sensitivity in the model. 
#' Parameters whose distributions are tightly confined around zero (the baseline value) are highly sensitive.
#'
#' For some parameters, in particular the preference parameters, their credible distributions may span a wider range than the inputs. This may seem
#' unexpected, but arises because within each feeding guild the preference parameters are not independent of each other. The preferences within each guild
#' must sum to 1. Hence, during the Monte Carlo process, after drawing new values of the preference values they are all rescaled to sum to 1, which may mean that
#' some of them will have been varied by more than the original coefficient of variation of the input random uniform. 
#'
#' For details of how the distribution of credible output values from StrathE2E are calculated see the help information for the e2ep_run_mc() function.
#'
#' @param model R-list object defining the model configuration, compiled by the e2ep_read() function
#' @param use.example Logical. If TRUE use pre-computed example data from the internal North Sea model rather than user-generated data (default=FALSE).
#'
#' @return Graphical display in a new graphics window.
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_mc_parameter_likelihoods <- function(model, use.example=FALSE) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	resultsdir	<- elt(model, "setup", "resultsdir")
	model.ident	<- elt(model, "setup", "model.ident")
	model.path	<- elt(model, "setup", "model.path")
	model.name 	<- elt(model, "setup", "model.name")
	model.variant 	<- elt(model, "setup", "model.variant")

corefilename   <-"CredInt_processed_parameters"
corefilename_f <-"CredInt_processed_parameters_f"
rawcorefilename<-"CredInt_cumulative_parameters"

if(use.example==TRUE){
	O_results   <- get.example.results(model.name, model.variant, corefilename, CREDINT_DIR)
	O_results_f <- get.example.results(model.name, model.variant, corefilename_f, CREDINT_DIR)
	Raw_results <- get.example.results(model.name, model.variant, rawcorefilename, CREDINT_DIR)
}

if(use.example==FALSE){
	credpath	<- makepath(resultsdir, CREDINT_DIR)
	credfile	<- csvname(credpath, corefilename, model.ident)
	credfile_f	<- csvname(credpath, corefilename_f, model.ident)
	rawcredfile	<- csvname(credpath, rawcorefilename, model.ident)
	message("Reading credible interval processed data from '", credfile, "'")

	O_results   <- readcsv(credfile, row.names=1)
	O_results_f <- readcsv(credfile_f, row.names=1)
	Raw_results <- readcsv(rawcredfile)
}

# The data from the file with the _f extension contains the results for the fixed parameters

	Nind   <-ncol(O_results)
	Nind_f <-ncol(O_results_f)
	Nind_all <- Nind+Nind_f

	annealing.parms <- read_SD_credint(model.path)	     # read the SD control file for Monte Carlo runs
        CVused<-annealing.parms$Prefsd                       # Pick up the CV for the preference values and assume all the others are the same

        N_iter<-nrow(Raw_results)                            # Extract the number of iterations used in the Monte Carlo run
        rm(Raw_results)                                      # discard the raw data

#Express the data as differences from the maxliklihood value

O_results_dif<-O_results
for(jj in 1:ncol(O_results)){
if(O_results[1,jj]==0){
	O_results_dif[1:6,jj]<-0
} else {
	O_results_dif[1:6,jj]<-(O_results[1:6,jj]-O_results[1,jj])/O_results[1,jj]
}
}

O_results_f_dif<-O_results_f
for(jj in 1:ncol(O_results_f)){
if(O_results_f[1,jj]==0){
	O_results_f_dif[1:6,jj]<-0
} else {
	O_results_f_dif[1:6,jj]<-(O_results_f[1:6,jj]-O_results_f[1,jj])/O_results_f[1,jj]
}
}

#Concatenate the fitted and fixed parameter results intoi a single dataframe

O_results_all_dif <- cbind(O_results_dif,O_results_f_dif)

#Colour vector for the box and whisker plots
bwcol<-c("white",rep("blue",ncol(O_results_dif)),rep("black",ncol(O_results_f_dif)))
prcol<-c("red",rep("white",ncol(O_results_all_dif)))

#bxpstats for the prior....

   array2plot<- array(dim=c(5,1),quantile(runif(N_iter,-CVused,+CVused),probs=c(0.005,0.25,0.5,0.75,0.995)))
   colnames(array2plot)<-"prior"
   bxpdata_prior<-list(stats=array2plot,n=rep(N_iter,1),conf=NULL,out=numeric(length=0))


#bpxstats for the posteriors

   modeldata2plot<-(O_results_all_dif[2:6,1])
   for(jj in 2:Nind_all) { modeldata2plot<-c(modeldata2plot,(O_results_all_dif[2:6,jj]))}
   array2plot<- array(dim=c(5,Nind_all),modeldata2plot)
   colnames(array2plot)<-colnames(O_results_all_dif)
   bxpdata_O<-list(stats=array2plot,n=rep(N_iter,Nind_all),conf=NULL,out=numeric(length=0),names=colnames(O_results_all_dif))


   modeldata2plot<-rep(0,5)
   for(jj in 1:Nind_all) { modeldata2plot<-c(modeldata2plot,(O_results_all_dif[2:6,jj]))}
   array2plot<- array(dim=c(5,Nind_all+1),modeldata2plot)
   colnames(array2plot)<-c("prior",colnames(O_results_all_dif))
   bxpdata_O<-list(stats=array2plot,n=rep(N_iter,Nind_all+1),conf=NULL,out=numeric(length=0),names=c("prior",colnames(O_results_all_dif)))

   modeldata2plot<-quantile(runif(N_iter,-CVused,+CVused),probs=c(0.005,0.25,0.5,0.75,0.995))
   for(jj in 1:Nind_all) { modeldata2plot<-c(modeldata2plot,rep(0,5))}
   array2plot<- array(dim=c(5,Nind_all+1),modeldata2plot)
   colnames(array2plot)<-c("prior",colnames(O_results_all_dif))
   bxpdata_prior<-list(stats=array2plot,n=rep(N_iter,Nind_all+1),conf=NULL,out=numeric(length=0),names=c("prior",colnames(O_results_all_dif)))

par(mar=c(3,2,1,1))
   bxp(bxpdata_prior,boxwex=0.35,at=seq(0,Nind_all),horizontal=TRUE,show.names=FALSE,las=1,ylim=c(-0.2,0.2),boxcol=prcol,whiskcol=prcol,whisklty="solid",medcol=prcol,staplecol=prcol)
   bxp(bxpdata_O,add=TRUE,boxwex=0.35,at=seq(0,Nind_all),yaxt="n",xaxt="n",cex.axis=1.1,horizontal=TRUE,boxcol=bwcol,whiskcol=bwcol,whisklty="solid",medcol=bwcol,staplecol=bwcol)
axis(side=2,las=1,at=seq(0,Nind_all),cex.axis=0.3)
mtext("Parameters relative to baseline values",cex=1.1,side=1,line=2)
abline(v=0,col="black")
abline(v=-0.15/2,col="red")
abline(v=+0.15/2,col="red")


}

