#
# plot_for_each_guild_catch_by_gear.R
#
#' Plots of the distributions of annual integrated catches of each guild across gears for the final year of a model run
#'
#' Multi-panel bar-plots of the distributions of annual integrated catches of each guild across gears for the final year of a model run generated by the e2ep_run() function
#'
#' The function plots a multi-panel page of bar-plots of annual integrated catches
#' of the guilds in the ecology model which are potentially catchable by the gears in the fishing fleet model:
#' macrophytes, carnivorous zooplankton, suspension/deposit and carnivore/scavenge feeding benthos, planktivorous, demersal and migratory fish,
#' birds, pinnipeds and cetaceans.
#'
#' Each panel represesents a single guild. In each panel, the x-axis represents the (up to) 12 gear types parameterised in the fishing fleet model, and the y-axis is the annual integrated catch.
#' For each gear there are two bars corresponding to the inshore and offshore zones. Each bar is subdivided into landed and discarded components of the catch.
#' Green and blue sections are offshore and inshore landings respectively; black and grey sections are offshore and inshore discards respectively.
#'
#' Units for the catch are mMN/m2/year
#'
#' @param model R-list object defining the model configuration compiled by the e2ep_read() function
#' @param results R-list object containing model results generated by the e2ep_run() function
#'
#' @return Graphical display in a new graphics window
#'
#' @importFrom graphics barplot title grconvertX grconvertY
#'
#' @noRd
#
# ------------------------------------------------------------------------------

plot_for_each_guild_catch_by_gear <- function(model, results) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	data			<- elt(model, "data")
	fleet.model		<- elt(data, "fleet.model")
	gear_codes		<- elt(fleet.model, "gear_codes")
	
	final.year.outputs	<- elt(results, "final.year.outputs")
	offshore_catchmat	<- elt(final.year.outputs, "offshore_catchmat")
	offshore_discmat	<- elt(final.year.outputs, "offshore_discmat")
	offshore_landmat	<- elt(final.year.outputs, "offshore_landmat")

	inshore_catchmat	<- elt(final.year.outputs, "inshore_catchmat")
	inshore_discmat		<- elt(final.year.outputs, "inshore_discmat")
	inshore_landmat		<- elt(final.year.outputs, "inshore_landmat")

	#----------------------------------------------------------------
	#In each plot panel:
	#    black and grey are offshore and inshore discards respectively
	#    green and blue are offshore and inshore landings respectively
	#----------------------------------------------------------------
	ngroups <- 10

	par(mfrow=c(6,2))
	par(mar=c(4,5,1,0.5))

	for(dsa in 1:(ngroups+1)) {

		mt<-(rownames(offshore_catchmat))[dsa]

		offshore_data2plot<-rbind(offshore_discmat[dsa,],offshore_landmat[dsa,])
		inshore_data2plot<-rbind(inshore_discmat[dsa,],inshore_landmat[dsa,])

		#colnames(offshore_data2plot)<-c("PT","PS","LLm","BT","DS","OT","GN","ST","NT","CR","MD","--")
		colnames(offshore_data2plot)<-gear_codes

		if(sum(offshore_data2plot+inshore_data2plot)>0) {
			yaxmax<-(1.2*max(c(offshore_catchmat[dsa,],inshore_catchmat[dsa,])))
		}

		if(sum(offshore_data2plot+inshore_data2plot)==0) {
			yaxmax<-1
		}

		barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,yaxmax),xlim=c(0,12),width=rep(0.4,12), space=c(0.5,rep(1.2,11)),yaxt="n",ann=FALSE,cex.axis=0.6)
		axis(side=2,las=1,cex.axis=0.9)
		if(dsa==1)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==3)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==5)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==7)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==9)  mtext("Catch",cex=0.7,side=2,line=3.5)
		if(dsa==11)  mtext("Catch",cex=0.7,side=2,line=3.5)
		title(main=mt,cex.main=1.0)
		barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,12), space=c(1.5,rep(1.2,11)),yaxt="n",xaxt="n",ann=FALSE)

	}

	#Calculate total catch aand discard by gear
	mt<-"All guilds combined"

	offshore_data2plot<-rbind(colSums(offshore_discmat),colSums(offshore_landmat))
	inshore_data2plot<-rbind(colSums(inshore_discmat),colSums(inshore_landmat))
	colnames(offshore_data2plot)<-gear_codes

	if(sum(offshore_data2plot+inshore_data2plot)>0) {
		yaxmax<-(1.2*max(c(colSums(offshore_catchmat),colSums(inshore_catchmat))))
	}

	if(sum(offshore_data2plot+inshore_data2plot)==0) {
		yaxmax<-1
	}

	barplot(offshore_data2plot,col=c("black","green"),ylim=c(0,yaxmax),xlim=c(0,12),width=rep(0.4,12), space=c(0.5,rep(1.2,11)),yaxt="n",ann=FALSE,cex.axis=0.5)
	axis(side=2,las=1,cex.axis=0.9)
	if(dsa==1)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==3)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==5)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==7)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==9)  mtext("Catch",cex=0.7,side=2,line=3.5)
	if(dsa==11)  mtext("Catch",cex=0.7,side=2,line=3.5)
	title(main=mt,cex.main=1.0)
	barplot(inshore_data2plot,col=c("grey","blue"),add=T,width=rep(0.4,12), space=c(1.5,rep(1.2,11)),yaxt="n",xaxt="n",ann=FALSE)
	legend(grconvertX(0.15, "ndc", "user"), grconvertY(0.05, "ndc", "user"),
	c( "offshore landings","offshore discards","inshore landings","inshore discards"),
	fill = c("green","black","blue","grey"), ncol=4, bty="n", xpd = NA)


}

