% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2ep_compare_runs_bar.R
\name{e2ep_compare_runs_bar}
\alias{e2ep_compare_runs_bar}
\title{Create tornado bar-plots of the differences between two model runs.}
\usage{
e2ep_compare_runs_bar(
  selection = "AAM",
  model1 = NA,
  use.saved1 = FALSE,
  results1,
  model2 = NA,
  use.saved2 = FALSE,
  results2,
  log.pc = "PC",
  zone = "W",
  bpmin = (-50),
  bpmax = (+50),
  maintitle = ""
)
}
\arguments{
\item{selection}{Text string from a list identifying which type of data are to be plotted. Select from: "AAM", "CATCH", corresponding to annual average mass data, and catch data respectively (default = "AAM"). Remember to include the phrase within "" quotes.}

\item{model1}{R-list object defining the baseline model configuration compiled by the e2ep_read() function.}

\item{use.saved1}{Logical. If TRUE then use baseline data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).}

\item{results1}{R-list object of baseline model output generated by the e2ep_run() (default=NULL).}

\item{model2}{R-list object defining the scenario model configuration compiled by the e2ep_read() function.}

\item{use.saved2}{Logical. If TRUE then use scenario data from a prior user-defined run held as csv files data in the current results folder (default=FALSE).}

\item{results2}{R-list object of scenario model output generated by the e2ep_run() (default=NULL).}

\item{log.pc}{Value="LG" for data to be plotted on a log10 scale, value = "PC" for data to be plotted on a percentage difference scale (default = "PC").}

\item{zone}{Value = "O" for offshore, "I" for inshore, or "W" for whole model domain (all upper case) (default = "W").}

\item{bpmin}{Axis minimum for plot - i.e. the maximum NEGATIVE value of (scenario-baseline). Default = -50, given log.pc="PC" (percentage differences). Needs to be reset to e.g. -0.3 if log.pc="LG" (log scale).}

\item{bpmax}{Axis maximum for plot - i.e. the maximum POSITIVE value of (scenario-baseline). Default = +50, given log.pc="PC" (percentage differences). Needs to be reset to e.g. +0.3 if log.pc="LG" (log scale).}

\item{maintitle}{A optional descriptive text field (in quotes) to be added above the plot. Keep to 45 characters including spaces (default="").}
}
\value{
List object comprising two dataframes of the displayed data, graphical display in a new graphics window.
}
\description{
Create a tornado bar-plot diagram of the differences in either annual average masses of ecology model varibles, or annual fishery catches, between two
different runs of the StrathE2E model, referred to as baseline and sceanrio runs.
}
\details{
A tornado plot is a horizontal barplot which shows the difference between baseline and sceanrio runs. Bars to the right indicate scenario values greater than the baseline, bars to the left indicate scenario values less than baseline.
In this function the difference between scenario and baseline data is calculated as either ((scenario - baseline)/baseline) and plotted on a linear (percentage) scale, or (scenario/baseline) and plotted on a log10 scale. In both cases, zero represents scenario = baseline.

The choice of whether to plot data on the masses of ecosystem components, or fishery catches, is determined by an argument setting in the function call. In either case two panels of tornado diagrams are plotted. In the case of mass data, the panels show
data on water column components of the ecoststem (upper panel) and seabed components of the system (lower panel). In the case of catch data, the upper panel shows landings, the lower shows discards. For mass data and landings, positive values (scenario > baseline) are
indicated by green bars to the right, negative values (scenario < baseline) by red bars to the left. For discards, positive values are in black, negative in grey. For any bars extending outsided the selected plotting range, the numeric value is displayed inside the relevant bar.

The function accepts data from either saved csv files created by the e2ep_run() function, or directly from e2ep_run() results objects.
The relevant csv files are: /results/Modelname/Variantname/ZONE_model_anav_biomass-*.csv where ZONE is INSHORE, OFFSHORE or WHOLEDOMAIN 
and * represents the model run identifier (model.ident) embedded in the R-list object created by the e2ep_read() function.

In addition to generating graphics output the function returns a list object of the data presented in the plot. The object comprises two dataframes (changewater and changeseabed where annual average mass data are selected; changeland and changedisc where catch data are selected).
The first column in each dataframe is the proportional difference expressed on a log10 scale, the second column as a percentage.
}
\examples{
# Load the 2011-2019 version of the internal Barents Sea model and run for 1 year
    m1 <- e2ep_read("Barents_Sea", "2011-2019",model.ident="11-19")
    r1 <-e2ep_run(m1,nyears=1)

# Load the 2011-2019 version of the internal Barents Sea model, modify, and run for 1 year
    m2 <- e2ep_read("Barents_Sea", "2011-2019",model.ident="plus1C")
    m2$data$physics.drivers$so_temp<-m2$data$physics.drivers$so_temp+1
# 1-degC added to offshore zone upper layer temperatures
    r2 <-e2ep_run(m2,nyears=1)

# Compare the annual average mass in 2011-2019 (as the baseline case) with "plus1C"
# (as the scenario case): 
    mdiff_results1 <- e2ep_compare_runs_bar(selection="AAM",
                                      model1=NA, use.saved1=FALSE, results1=r1,
			         model2=NA, use.saved2=FALSE, results2=r2,
	                         log.pc="PC", zone="W",
                 		         bpmin=(-50),bpmax=(+50),
      		                 maintitle="2011-2019 compared with 1C warmer")
    mdiff_results1

# Compare the annual catch in 2011-2019 (as the baseline case) with "plus1C"
# (as the scenario case): 
    mdiff_results2 <- e2ep_compare_runs_bar(selection="CATCH",
                                      model1=NA, use.saved1=FALSE, results1=r1,
			         model2=NA, use.saved2=FALSE, results2=r2,
	                         log.pc="LG", zone="W",
                     	         bpmin=(-0.9),bpmax=(+0.4),
      		                 maintitle="2011-2019 compared with 1C warmer")
    mdiff_results2

\donttest{
# Create a new scenario run from 2011-2019 case. 
# Copy the 2011-2019 configuration into a new model object:
    scen1_model   <- m1
    scen1_model$setup$model.ident <- "scenario1"
# Gear 4 (Beam_Trawl_BT1+BT2) activity rate rescaled to 0.5*baseline:
    scen1_model$data$fleet.model$gear_mult[4] <- 0.5
    scen1_results <- e2ep_run(scen1_model,nyears=20)

# Compare the annual average mass from the the 2011-2019 baseline with scenario1 data 
    mdiff_results3 <- e2ep_compare_runs_bar(selection="AAM",
                                 model1=NA, use.saved1=FALSE, results1=r1,
                                 model2=NA, use.saved2=FALSE, results2=scen1_results,
                                 log.pc="PC", zone="W",
                                 bpmin=(-30),bpmax=(+30),
                                 maintitle="Beam Trawl activity reduced by half")
    mdiff_results3
}

\donttest{
# Create a second sceanario from the 2011-2019 case, this time saving the results to a file;
# csv output to temporary folder since results.path was not set in e2ep_read() when creating m1.
# Copy the baseline configuration into a new model object:
    scen2_model   <- m1
    scen2_model$setup$model.ident <- "scenario2"
# Gear 1 (Pelagic_Trawl+Seine) activity rate rescaled to 0.5*baseline:
    scen2_model$data$fleet.model$gear_mult[1] <- 0.5
    scen2_results <- e2ep_run(scen2_model,nyears=20, csv.output=TRUE)

# Compare the annual catches in the 2011-2019 base line with the Pelagic Trawl/seine scenario
    mdiff_results4 <- e2ep_compare_runs_bar(selection="CATCH",
                                 model1=NA, use.saved1=FALSE, results1=r1,
                     	    model2=scen2_model,use.saved2=TRUE, results2=NA,
                     	    log.pc="LG", zone="W",
                     	    bpmin=(-0.4),bpmax=(+0.6),
                     	    maintitle="Pelagic Trawl/Seine activity reduced by half")
    mdiff_results4
}
}
\seealso{
\code{\link{e2ep_read}}, \code{\link{e2ep_run}}
}
