% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/e2ep_read.R
\name{e2ep_read}
\alias{e2ep_read}
\title{Read all the configuration, driving and parameter files for a designated model and compile the data into a list object.}
\usage{
e2ep_read(
  model.name,
  model.variant,
  models.path = NULL,
  results.path = NULL,
  results.subdir = "",
  model.ident = "base",
  quiet = TRUE,
  silent = FALSE
)
}
\arguments{
\item{model.name}{Name of model to read (no default).}

\item{model.variant}{Variant of the model to be read (no default).}

\item{models.path}{Relative path from the current working directory to a folder containing a library of model configurations (typically "Folder/Models"). Setting models.path="" is valid. Default models.path=NULL, meaning read a Barents Sea model setup from the package folder extdata/Models.}

\item{results.path}{Relative path from the current working directory to a folder for writing and reading model output files (e.g. "E2E_results"). Setting results.path="" is valid. Model-specific sub-folders will be assigned and if necessary auto-created according to the model name and variant. Default results.path=NULL, meaning write/read to/from a temporary directory.}

\item{results.subdir}{Subdirectory of "working_directory/results.path/model_name/model_variant" to be created if required. (Default="", meaning no subdirectory will be created).}

\item{model.ident}{Identifier text appended to output file names (e.g. OFFSHORE_model_annualresults-TEXT.csv instead of just OFFSHORE_model_annualresults.csv). (Default="base").}

\item{quiet}{Logical. If FALSE then see on-screen information on indvidual parameter files as they are read (default=TRUE).}

\item{silent}{Logical. If FALSE then see on-screen information on model and results paths (default=FALSE). If set TRUE then this over-rides any quiet=FALSE setting and forces quiet=TRUE.}
}
\value{
R-list object containing all of the model configuration data.
}
\description{
This function is the key point of entry to the package. It reads all the csv files containing the configuration, driving and parameter values which define
a model for a particular region and compiles them into a single R-list object. This data object forms the basis for almost all other functions in the package.
}
\details{
The input csv files are specified in the MODEL_SETUP.csv file located
in the Model/variant folder specified by the argument models.path in a e2ep_read() call, or from the extdata/Models folder
in the package installation. The model supplied with the package has the variant named 2011-2019 and is a fully
parameterised and documented model of the Barents Sea.

Starting from a baseline model configuration defined in the MODEL_SETUP.csv file, any of the terms in the R-list objcet created by e2ep_read() can be modified by coding
to produce an unlimited range of scenario configurations representing, for example, changes in the physical environment, changes in the composition or activity of the
fishing fleets, or any combination of these. The power of this approach is that large numbers of scenarios can be coded in standard R and simulated without any need
for manual editing of input files.
}
\examples{
# Load the 2011-2019 version of the Barents Sea model supplied with the package; outputs go
# to a temporary results folder; default model.ident setting.
# Default screen information settings - on-screen parameter file information suppressed but  
# path information enabled:
    model <- e2ep_read("Barents_Sea", "2011-2019")

# User-specified model.ident; on-screen parameter file and path information enabaled:
    model <- e2ep_read("Barents_Sea", "2011-2019",model.ident="baseline",quiet=FALSE)

# All on-screen information suppressed even though quiet=FALSE:
    model <- e2ep_read("Barents_Sea", "2011-2019",model.ident="baseline",quiet=FALSE,silent=TRUE)

# View details of the structure and contents of the model list-object:
    str(model,max.level=1)
    str(model,max.level=2)

# Dummy example to illustrate loading a user defined model from a user workspace
# ... substitute your own path details for e.g. "Folder/Models":
#     model<-e2ep_read("Modelname", "Variantname",
#                      models.path="Folder/Models",
#                      model.ident="demo")

# Dummy example to illustrate loading a user defined model from a user workspace and
# sending the model outputs to a specified folder ... substitute your own path details
# for the dummy paths shown here 
# for "Folder/Models" and "Folder/results":
#     model<-e2ep_read("Modelname", "Variantname",
#                      models.path="Folder/Models",
#                      results.path="Folder/results",
#                      model.ident="demo")

# Create a new scenario version of the Barents Sea/2011-2019 model by increasing the
# activity rate of gear 1 (pelagic trawls and seines) by a factor of 2:
    model <- e2ep_read("Barents_Sea", "2011-2019")
    scenario_model <- model
    scenario_model$data$fleet.model$gear_mult[1] <- 2
# Set a new identifier for any csv outputs:
    scenario_model$setup$model.ident <- "gear1x2"

# Run the baseline model for 2 years:
    results<-e2ep_run(model,nyears=2)
# Visualise the time series of output from the baseline model run:
    e2ep_plot_ts(model,results,selection="ECO")

\donttest{
# Run the scenario model for 20 years:
    scenario_results<-e2ep_run(scenario_model,nyears=20)
# Visualise the time series of output from the scenario model run
# in a new graphics window:
    dev.new()
    e2ep_plot_ts(scenario_model,scenario_results,selection="ECO")
}


}
\seealso{
\code{\link{e2ep_ls}}, \code{\link{e2ep_copy}}, \code{\link{e2ep_run}}
}
