#
# boxplot_monthly_compare_runs.R
#
#' Box-and-whisker plots of baseline and scenario simulated monthly averaged nutrient and plankton data with credible intervals derived from Monte Carlo StrathE2E runs.
#'
#' Creates a multi-panel plot comparing a range of simulated monthly averaged nutrient and plankton data from a baseline and a scenario mode run, with the distribution of credible values
#' generated by the e2ep_run_mc() Monte Carlo function.
#'
#' For details of how the distribution of credible output values from StrathE2E are calculated see help(e2ep_run_mc).
#'
#' The function plots a multi-panel page of box-and-whisker plots showing the medians and variability ranges (quartiles as box-and-whisker) of a range of monthly averaged nutrient and plankton data from the final year of a baseline run
#' (shown in black), alongside comparable box-and-whisker plots (shown in red) of the same measures derived from the final year of a scenario run. The credible intervals for each case need ot be generted by the Monte Carlo methodology (e2ep_run_mc() function).
#'
#' Optionally the function can read an example data set for one of the two North Sea model variants supplied with the package.
#'
#' @param model1 R-list object defining the baseline model configuration compiled by the e2ep_read() function.
#' @param ci.data1 Logical. If TRUE plot credible intervals around model results based on Monte Carlo simulation withteh e2ep_run_mc() function, (default=FALSE).
#' @param use.saved1 Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder, (default=FALSE).
#' @param use.example1 Logical. If TRUE use pre-computed example data from the internal North Sea model as the baseline rather than user-generated data, (default=FALSE).
#' @param results1 R-list object of baseline model output generated by the e2ep_run(), (default=NULL).
#' @param model2 R-list object defining the baseline model configuration compiled by the e2ep_read() function.
#' @param ci.data2 Logical. If TRUE plot credible intervals around model results based on Monte Carlo simulation withteh e2ep_run_mc() function, (default=FALSE).
#' @param use.saved2 Logical. If TRUE use data from a prior user-defined run held as csv files data in the current results folder, (default=FALSE).
#' @param use.example2 Logical. If TRUE use pre-computed example data from the internal North Sea model as the scenario rather than user-generated data, (default=FALSE).
#' @param results2 R-list object of scenario model output generated by the e2ep_run(), (default=NULL).
#'
#' @return Graphical display in a new graphics window.
#'
#' @noRd
#
# ---------------------------------------------------------------------
# |                                                                   |
# | Authors: Mike Heath, Ian Thurlbeck                                |
# | Department of Mathematics and Statistics                          |
# | University of Strathclyde, Glasgow                                |
# |                                                                   |
# | Date of this version: May 2020                                    |
# |                                                                   |
# ---------------------------------------------------------------------

boxplot_monthly_compare_runs <- function(model1, ci.data1=FALSE, use.saved1=FALSE, use.example1=FALSE, results1=NULL,
                                         model2, ci.data2=FALSE, use.saved2=FALSE, use.example2=FALSE, results2=NULL) {

start_par = par()$mfrow
on.exit(par(mfrow = start_par))

	resultsdir1		<- elt(model1, "setup", "resultsdir")
	model.ident1		<- elt(model1, "setup", "model.ident")
	model.path1		<- elt(model1, "setup", "model.path")
	model.name1		<- elt(model1, "setup", "model.name")
	model.variant1 		<- elt(model1, "setup", "model.variant")

	resultsdir2		<- elt(model2, "setup", "resultsdir")
	model.ident2		<- elt(model2, "setup", "model.ident")
	model.path2		<- elt(model2, "setup", "model.path")
	model.name2		<- elt(model2, "setup", "model.name")
	model.variant2 		<- elt(model2, "setup", "model.variant")

	#Read the observed data file
	#Format expected = 7 columns
	#Month	Variable	median	lower_centile	upper_centile	Units	low_cent_value	upp_cent_value	Comments
	#The variable names expected are:
	#surface_nitrate
	#deep_nitrate
	#surface_ammonia
	#deep_ammonia
	#surface_chlorophyll
	#omniv_zooplankton
	#carniv_zooplankton
	#larvae_susp_dep_benthos
	#larvae_carn_scav_benthos
#	obstargetdataset	<- get.model.file(model.path, TARGET_DATA_DIR, file.pattern=MONTHLY_TARGET_DATA)


	corefilename<-"CredInt_processed_monthly_mass"


	monlab<-c("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec")


if(ci.data1==TRUE){

if(use.example1==TRUE){
	credintervaldata1 <- get.example.results(model.name1, model.variant1, corefilename, CREDINT_DIR)
}

if(use.example1==FALSE){
	credpath1	<- makepath(resultsdir1, CREDINT_DIR)
	credfile1	<- csvname(credpath1, corefilename, model.ident1)
	if (! file.exists(credfile1)) {
		message("Error: cannot find credible interval output file: ", credfile1)
		stop("Please run the Monte Carlo function!\n")
	}
	message("Reading credible interval processed data from '", credfile1, "'")
	credintervaldata1	<- readcsv(credfile1, row.names=1)	# first column is row names
}


}


if(ci.data2==TRUE){

if(use.example2==TRUE){
	credintervaldata2 <- get.example.results(model.name2, model.variant2, corefilename, CREDINT_DIR)
}

if(use.example2==FALSE){
	credpath2	<- makepath(resultsdir2, CREDINT_DIR)
	credfile2	<- csvname(credpath2, corefilename, model.ident2)
	if (! file.exists(credfile2)) {
		message("Error: cannot find credible interval output file: ", credfile2)
		stop("Please run the Monte Carlo function!\n")
	}
	message("Reading credible interval processed data from '", credfile2, "'")
	credintervaldata2	<- readcsv(credfile2, row.names=1)	# first column is row names
}

}


if(ci.data1==FALSE){

if(use.saved1==TRUE){
	datafile1	<- csvname(resultsdir1, "model_monthlyresults", model.ident1)
	print(paste("Using baseline data held in a file ",datafile1," from a past model run"))
	check.exists(datafile1)
	modelmonthly1<-readcsv(datafile1)
}

if(use.saved1==FALSE){
	modelmonthly1	<- elt(results1, "final.year.outputs", "monthly.averages")
}

}


if(ci.data2==FALSE){

if(use.saved2==TRUE){
	datafile2	<- csvname(resultsdir2, "model_monthlyresults", model.ident2)
	print(paste("Using scenario data held in a file ",datafile2," from a past model run"))
	check.exists(datafile2)
	modelmonthly2<-readcsv(datafile2)
}

if(use.saved2==FALSE){
	modelmonthly2	<- elt(results2, "final.year.outputs", "monthly.averages")
}

}

# --------------------------------------------------------------------------


if(ci.data1==FALSE){
    #convert modelmonthly into credintervaldata format
	credintervaldata1 <- data.frame(rep(rep(NA,6*ncol(modelmonthly1))))
	for(jj in 2:12){
		credintervaldata1[,jj]<-rep(rep(NA,6*ncol(modelmonthly1)))
	}
	colnames(credintervaldata1)<-c("1","2","3","4","5","6","7","8","9","10","11","12")

	for(jj in 1:ncol(modelmonthly1)){
		nameset<-c(paste((names(modelmonthly1))[jj],"-maxlik",sep=""),
			   paste((names(modelmonthly1))[jj],"-0.005",sep=""),
			   paste((names(modelmonthly1))[jj],"-0.25",sep=""),
			   paste((names(modelmonthly1))[jj],"-0.5",sep=""),
			   paste((names(modelmonthly1))[jj],"-0.75",sep=""),
			   paste((names(modelmonthly1))[jj],"-0.995",sep="") )
		if(jj==1) fullnameset<-nameset
		if(jj>1)  fullnameset<-c(fullnameset,nameset)
	}
		rownames(credintervaldata1)<-fullnameset
	for(jj in 1:ncol(modelmonthly1)){
		for(kk in 1:6){
		credintervaldata1[(((jj-1)*6)+kk),1:12] <- as.numeric(modelmonthly1[,jj])
	}
	}
}

if(ci.data2==FALSE){
    #convert modelmonthly into credintervaldata format
	credintervaldata2 <- data.frame(rep(rep(NA,6*ncol(modelmonthly2))))
	for(jj in 2:12){
		credintervaldata2[,jj]<-rep(rep(NA,6*ncol(modelmonthly2)))
	}
	colnames(credintervaldata2)<-c("1","2","3","4","5","6","7","8","9","10","11","12")

	for(jj in 1:ncol(modelmonthly2)){
		nameset<-c(paste((names(modelmonthly2))[jj],"-maxlik",sep=""),
			   paste((names(modelmonthly2))[jj],"-0.005",sep=""),
			   paste((names(modelmonthly2))[jj],"-0.25",sep=""),
			   paste((names(modelmonthly2))[jj],"-0.5",sep=""),
			   paste((names(modelmonthly2))[jj],"-0.75",sep=""),
			   paste((names(modelmonthly2))[jj],"-0.995",sep="") )
		if(jj==1) fullnameset<-nameset
		if(jj>1)  fullnameset<-c(fullnameset,nameset)
	}
		rownames(credintervaldata2)<-fullnameset
	for(jj in 1:ncol(modelmonthly2)){
		for(kk in 1:6){
		credintervaldata2[(((jj-1)*6)+kk),1:12] <- as.numeric(modelmonthly2[,jj])
	}
	}
}





#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	# BASELINE MODEL DATA:
	#Generate the list objects needed by the bxp plotting function
	for(iii in 1:9){

		credrows<- seq(   ((iii-1)*(5+1))+2,((iii-1)*(5+1))+(5+1)   )

		modeldata2plot<-(credintervaldata1[credrows,1])
		for(jj in 2:12) { modeldata2plot<-c(modeldata2plot,(credintervaldata1[credrows,jj]))}

		array2plot<- array(dim=c(5,12),modeldata2plot)
		bxpdata<-list(stats=array2plot,n=rep(100,12),conf=NULL,out=numeric(length=0),names=monlab)
		# bxp(bxpdata,boxwex=0.3,at=seq(1,12)+0.35,xlim=c(0,13),ylim=c(0,max(modeldata2plot)*1.1))
		if(iii==1) bxpdata1<-bxpdata
		if(iii==2) bxpdata2<-bxpdata
		if(iii==3) bxpdata3<-bxpdata
		if(iii==4) bxpdata4<-bxpdata
		if(iii==5) bxpdata5<-bxpdata
		if(iii==6) bxpdata6<-bxpdata
		if(iii==7) bxpdata7<-bxpdata
		if(iii==8) bxpdata8<-bxpdata
		if(iii==9) bxpdata9<-bxpdata
	}

	bxpdata10<-list(stats=(bxpdata9$stats+bxpdata8$stats),n=rep(100,12),conf=NULL,out=numeric(length=0),names=monlab)
	#Combines the two types of benthic larvae

	#Package all the bxpdata objects up into a list t pass into the plotting function
	bxpdata.all.1<-list(bxpdata1=bxpdata1,
			  bxpdata2=bxpdata2,
			  bxpdata3=bxpdata3,
			  bxpdata4=bxpdata4,
			  bxpdata5=bxpdata5,
			  bxpdata6=bxpdata6,
			  bxpdata7=bxpdata7,
			  bxpdata8=bxpdata8,
			  bxpdata9=bxpdata9,
			  bxpdata10=bxpdata10)



	# SCENARIO MODEL DATA:
	#Generate the list objects needed by the bxp plotting function
	for(iii in 1:9){

		credrows<- seq(   ((iii-1)*(5+1))+2,((iii-1)*(5+1))+(5+1)   )

		modeldata2plot<-(credintervaldata2[credrows,1])
		for(jj in 2:12) { modeldata2plot<-c(modeldata2plot,(credintervaldata2[credrows,jj]))}

		array2plot<- array(dim=c(5,12),modeldata2plot)
		bxpdata<-list(stats=array2plot,n=rep(100,12),conf=NULL,out=numeric(length=0),names=monlab)
		# bxp(bxpdata,boxwex=0.3,at=seq(1,12)+0.35,xlim=c(0,13),ylim=c(0,max(modeldata2plot)*1.1))
		if(iii==1) bxpdata1<-bxpdata
		if(iii==2) bxpdata2<-bxpdata
		if(iii==3) bxpdata3<-bxpdata
		if(iii==4) bxpdata4<-bxpdata
		if(iii==5) bxpdata5<-bxpdata
		if(iii==6) bxpdata6<-bxpdata
		if(iii==7) bxpdata7<-bxpdata
		if(iii==8) bxpdata8<-bxpdata
		if(iii==9) bxpdata9<-bxpdata
	}

	bxpdata10<-list(stats=(bxpdata9$stats+bxpdata8$stats),n=rep(100,12),conf=NULL,out=numeric(length=0),names=monlab)
	#Combines the two types of benthic larvae

	#Package all the bxpdata objects up into a list t pass into the plotting function
	bxpdata.all.2<-list(bxpdata1=bxpdata1,
			  bxpdata2=bxpdata2,
			  bxpdata3=bxpdata3,
			  bxpdata4=bxpdata4,
			  bxpdata5=bxpdata5,
			  bxpdata6=bxpdata6,
			  bxpdata7=bxpdata7,
			  bxpdata8=bxpdata8,
			  bxpdata9=bxpdata9,
			  bxpdata10=bxpdata10)


#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	plotdata_mco<-function(bxpdata.all.1, bxpdata.all.2, obspar, monlab){

	if(obspar==1){
		bxpdata1<-bxpdata.all.1$bxpdata1
		bxpdata2<-bxpdata.all.2$bxpdata1
	}
	if(obspar==2){
		bxpdata1<-bxpdata.all.1$bxpdata2
		bxpdata2<-bxpdata.all.2$bxpdata2
	}
	if(obspar==3){
		bxpdata1<-bxpdata.all.1$bxpdata3
		bxpdata2<-bxpdata.all.2$bxpdata3
	}
	if(obspar==4){
		bxpdata1<-bxpdata.all.1$bxpdata4
		bxpdata2<-bxpdata.all.2$bxpdata4
	}
	if(obspar==5){
		bxpdata1<-bxpdata.all.1$bxpdata5
		bxpdata2<-bxpdata.all.2$bxpdata5
	}
	if(obspar==6){
		bxpdata1<-bxpdata.all.1$bxpdata6
		bxpdata2<-bxpdata.all.2$bxpdata6
	}
	if(obspar==7){
		bxpdata1<-bxpdata.all.1$bxpdata7
		bxpdata2<-bxpdata.all.2$bxpdata7
	}
	if(obspar==8){
		bxpdata1<-bxpdata.all.1$bxpdata8
		bxpdata2<-bxpdata.all.2$bxpdata8
	}
	if(obspar==9){
		bxpdata1<-bxpdata.all.1$bxpdata9
		bxpdata2<-bxpdata.all.2$bxpdata9
	}
	if(obspar==10){
		bxpdata1<-bxpdata.all.1$bxpdata10
		bxpdata2<-bxpdata.all.2$bxpdata10
	}

	modplot1<-bxpdata1$stats
	modplot2<-bxpdata2$stats

	if(obspar==1 | obspar==2) ymax<- max(0, max(as.data.frame(bxpdata.all.1$bxpdata1$stats)), max(as.data.frame(bxpdata.all.1$bxpdata2$stats)), max(as.data.frame(bxpdata.all.2$bxpdata1$stats)), max(as.data.frame(bxpdata.all.2$bxpdata2$stats)),na.rm=TRUE )
	if(obspar==3 | obspar==4) ymax<- max(0, max(as.data.frame(bxpdata.all.1$bxpdata3$stats)), max(as.data.frame(bxpdata.all.1$bxpdata4$stats)), max(as.data.frame(bxpdata.all.2$bxpdata3$stats)), max(as.data.frame(bxpdata.all.2$bxpdata4$stats)),na.rm=TRUE )

	if(obspar>4) ymax<- max(0, max(as.data.frame(modplot1),na.rm=TRUE),max(as.data.frame(modplot2),na.rm=TRUE),na.rm=TRUE )

	if(ymax==0 | is.na(ymax)==TRUE) ymax<-1

		bxp(bxpdata1,boxwex=0.25,at=1:12,yaxt="n",ylim=c(0,ymax*1.1),show.names=FALSE,las=1,cex.axis=1.1,
		boxcol="black",whiskcol="black",whisklty="solid",medcol="black",staplecol="black")

		axis(labels=monlab, at=seq(1,12),side=1,las=1,cex.axis=1.1,padj=-0.55)


		if(obspar==1){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Surf.nitrate",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==2){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Deep nitrate",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==3){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Surf.ammonia",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==4){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Deep ammonia",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==5){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Chlorophyll",cex=0.8,side=2,line=4)
			mtext(bquote(mg.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==6){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Omniv.zoo",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==7){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Carniv.zoo",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==8){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Larv.s/d.benth.",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==9){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Larv.c/s.benth.",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}
		
		if(obspar==10){
			axis(side=2,cex.lab=1.0,las=1)
			mtext("Benthos larvae (all)",cex=0.8,side=2,line=4)
			mtext(bquote(mMN.m^-3),cex=0.6,side=2,line=2.7)
		}


		bxp(bxpdata2,add=TRUE,boxwex=0.25,at=1:12+0.35,yaxt="n",xaxt="n",
		boxcol="red",whiskcol="red",whisklty="solid",medcol="red",staplecol="red")

	}
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



par(mfrow=c(4,2))

par(mar=c(3,6,0.6,0.5))


plotdata_mco(bxpdata.all.1, bxpdata.all.2, 1, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 2, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 3, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 4, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 5, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 6, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 7, monlab)

#plotdata_mco(bxpdata.all.1, bxpdata.all.2, 8, monlab)

#plotdata_mco(bxpdata.all.1, bxpdata.all.2, 9, monlab)

plotdata_mco(bxpdata.all.1, bxpdata.all.2, 10, monlab)

	legend(grconvertX(0.425, "ndc", "user"), grconvertY(0.045, "ndc", "user"),
	c("baseline","scenario"), fill = c("black","red"), ncol=2, bty="n", xpd = NA)

}
